﻿/** 
 * @file		DtFunc2000.h
 * @brief		Data Transfer/Resource Data Converter Application Program Interface Functions(GOT2000)
 *				[データ転送/リソースデータ変換API関数(GOT2000)]
 * @date		2014
 * Copyright	(C) 2008 MITSUBISHI Electric Corporation All Rights Reserved.
 */


#ifndef DT_FUNC2000_H___
#define DT_FUNC2000_H___


///GOT drive [GOTドライブ]
#define DTCOMM2000_GOT_DRIVE_NAME_A			L"A"
#define DTCOMM2000_GOT_DRIVE_NAME_B			L"B"
#define DTCOMM2000_GOT_DRIVE_NAME_C			L"C"
#define DTCOMM2000_GOT_DRIVE_NAME_D			L"D"
#define DTCOMM2000_GOT_DRIVE_NAME_E			L"E"
#define DTCOMM2000_GOT_DRIVE_NAME_F			L"F"
#define DTCOMM2000_GOT_DRIVE_NAME_G			L"G"

///Select communication configuration type [通信設定typeの指定]
#define DTCOMM2000_COMMCONFIG_TYPE_RS232C	L"RS232"
#define DTCOMM2000_COMMCONFIG_TYPE_USB		L"USB"
#define DTCOMM2000_COMMCONFIG_TYPE_ETHERNET	L"ETHERNET"

///Select communication configuration baudrate [通信設定baudrateの指定]
#define DTCOMM2000_COMMCONFIG_BAUDRATE_115200	L"115200"
#define DTCOMM2000_COMMCONFIG_BAUDRATE_57600	L"57600"
#define DTCOMM2000_COMMCONFIG_BAUDRATE_38400	L"38400"
#define DTCOMM2000_COMMCONFIG_BAUDRATE_19200	L"19200"
#define DTCOMM2000_COMMCONFIG_BAUDRATE_9600		L"9600"

///Select communication configuration plc [通信設定plcの指定]
#define DTCOMM2000_COMMCONFIG_PLC_RCPU		L"RCPU"
#define DTCOMM2000_COMMCONFIG_PLC_QCPU		L"QCPU"
#define DTCOMM2000_COMMCONFIG_PLC_LCPU		L"LCPU"
#define DTCOMM2000_COMMCONFIG_PLC_RJ71EN71	L"RJ71EN71"
#define DTCOMM2000_COMMCONFIG_PLC_QJ71E71	L"QJ71E71"
#define DTCOMM2000_COMMCONFIG_PLC_LJ71E71	L"LJ71E71"

///Select communication configuration pc_protocol [通信設定pc_protocolの指定]
#define DTCOMM2000_COMMCONFIG_PROTOCOL_TCP	L"TCP"
#define DTCOMM2000_COMMCONFIG_PROTOCOL_UDP	L"UDP"

///Select communication configuration plc_connect_type [通信設定plc_connect_typeの指定]
#define DTCOMM2000_COMMCONFIG_CONNECT_ETHERNET_PORT	L"EthernetPort"
#define DTCOMM2000_COMMCONFIG_CONNECT_VIA_HUB		L"ViaHub"

///Select communication configuration plc_connect_format/unit_connect_format [通信設定plc_connect_format/unit_connect_formatの指定]
#define DTCOMM2000_COMMCONFIG_FORMAT_IP		L"IP"
#define DTCOMM2000_COMMCONFIG_FORMAT_HOST	L"HostName"

///Select communication configuration unit_staion_ip_info [通信設定unit_staion_ip_infoの指定]
#define DTCOMM2000_COMMCONFIG_STATION_IP_INFO_AUTORESPONSE	L"AutoResponseSystem"
#define DTCOMM2000_COMMCONFIG_STATION_IP_INFO_OTHERS		L"Others"


///Data transfer [データ転送.]
///No communication error [通信エラーなし]
#define COMM2000_ERROR_N				0

///Operation error code [動作エラーコード]
enum EConsoleCommError2000{
	DTCOMM2000_NOERROR = 0,
	DTCOMM2000_OPTION_ERROR,		///When an incorrect command/option is specified, when the specified value is incorrect 
									///[不正なコマンド・オプションが指定されたとき、指定値が不正なとき]
	DTCOMM2000_TRANS_FILE_ERROR,	///When the transfer file is incorrect [転送ファイルが不正なとき]
	DTCOMM2000_TRANS_FILE_ERROR_SYSLABEL,///(GOT1000)
	DTCOMM2000_TRANS_INI_ERROR,		///(GOT1000)
	DTCOMM2000_TRANS_INI_DATA_ERROR,///When transfer specification is incorrect 
									///[転送指定が不正なとき]
	DTCOMM2000_NO_TRANS_DATA_ERROR,	///When transfer data does not exist [転送データが存在しないとき]
	DTCOMM2000_FILE_SAVE_ERROR,		///When file save fails (+ read-only) [ファイル保存に失敗したとき(+上書き禁止)]
	DTCOMM2000_COMM_INFO_ERROR,		///When an incorrect value is specified for communication setting 
									///[通信設定指定で不正値が指定されたとき]
	DTCOMM2000_PROJECT_SECURITY_ERROR,///(GOT1000)
	DTCOMM2000_COMM_CANCEL = 20,	///When an operation is canceled in a communication [通信で動作キャンセルされたとき]
	DTCOMM2000_COMM_OPEN_ERROR,		///When opening the communication port fails [通信ポートのオープンに失敗したとき]
	DTCOMM2000_COMM_ERROR,			///When an error occurs during communication [通信中にエラーが発生したとき]
	DTCOMM2000_COMM_NOTAUTHENTICATED,///Password authentication error [パスワード認証エラー]
	DTCOMM2000_COMM_NO_DRIVE,		///Drive specification error [ドライブ指定エラー]
	DTCOMM2000_COMM_ALREADY_EXISTS = 40,///Exclusive lock acquisition error [排他ロック取得エラー]
	DTCOMM2000_ERROR = 50			///Other errors [その他のエラー]
};


///Resource data conversion [リソースデータ変換]
///Convert complete [変換成功]
#define CONVERT2000_COMPLETE			 0
#define CONVERT2000_WITH_DEVCMT_NOTCNV	 1	///The device comments are not reflected since they were not included in the conversion target.
											///[デバイスコメントが変換対象になっていないため、デバイスコメントは反映されていない]

///Error code [エラーコード]
#define ERROR2000_INCORRECTFILE			-1  ///The file data is incorrect. [ファイルデータが不正]
#define ERROR2000_NO_FILE				-2	///The file does not exist. [ファイルが存在しない]
#define ERROR2000_NO_FILETYPE			-3	///Unsupported file type [対応するファイルタイプではない]
#define	ERROR2000_NO_LANGUAGETYPE		-4	///Unsupported language type [対応しない言語タイプ]
#define ERROR2000_NO_FILE_G2P			-5	///The destination G2P file does not exist. [変換先のG2Pファイルが存在しない]
#define ERROR2000_NO_FILE_OLCONVFILE	-6	///Data(Sap2000、G2SYSLANGINFO.INI) required for a conversion of operation logs/alarm does not exist. 
											///[操作ログ/アラームの変換に必要なデータ(Sap2000、G2SYSLANGINFO.INI)がない]
#define ERROR2000_NO_RCDLL_FILE			-7	///dll does not exist in the data transfer tool install folder. 
											///[データ転送ツールインストールフォルダに必要なdllが存在しない]
#define ERROR2000_NO_PROJECT			-8	///The project data does not exist. Invalid (broken) project data is specified.
											///[プロジェクトデータが存在しない。不正な(壊れている)プロジェクトデータを指定した。]
#define ERROR2000_INCORRECT_PASSWORD	-9	///The project user name or password is wrong. Or the password is not specified.
											///[プロジェクトのユーザ名、パスワードが違う。またはパスワードを指定していない。]
#define ERROR2000_INCORRECT_PROJECT		-10	///The setting of user alarm observation does not exist in the project data.
											///The required comment group setting does not exist in the project data.
											///The specified setting of comment column No. does not exist in the project data.
											///[プロジェクトデータにユーザアラーム監視の設定が存在しない。]
											///[プロジェクトデータに必要なコメントグループの設定が存在しない。]
											///[プロジェクトデータに指定したコメント列No.の設定が存在しない。]
#define ERROR2000_CANNOT_CREATE_LOG		-11	///The file after conversion cannot be written. Log file cannot be created.
											///(When targeting files in the same path)
											///[変換先のファイルを書込めない。ログファイルが作成できない。]
											///[(同じパス内のファイルを対象にする指定時)]
#define ERROR2000_CANNOT_USE_PROJECT	-12	///Required functions are not installed.
											///[必要な機能がインストールされていない。]
#define ERROR2000_SECURITY_KEY			-13	///The project data cannot be opened with the security key registered on the computer.
											///[プロジェクトデータがパソコンに登録されているセキュリティキーでは開けない。]


/** 
 * @brief	DT2000_Download()
 * @note	Write the project file. [プロジェクトファイルを書き込む。]
 *			Regard an empty text as being specified in NULL is specified for an argument. [引数にNULLを指定した場合、空の文字列が指定されたと見なす。]
 *			Display an error prior to transfer if the number of characters set for a password exceeds its maximum. [パスワードが最大文字数より大きい場合、転送前にエラーとする。]
 * @param	package_filename	Write package [書込パッケージ]
 * @param	got_drive_name		GOT drive [GOTドライブ]
 * @param	init_sram			Specification to initialize SRAM user area [SRAMユーザ領域初期化を行う指定]
 * @param	notreboot			Specification not to automatically restart after download [ダウンロード後自動で再起動しない指定]
 * @param	project_username	User Name of Project [プロジェクトのユーザ名]
 * @param	project_password	Password of Project [プロジェクトのパスワード]
 * @param	remote_password		Remote password [リモートパスワード]
 * @param	got_password		Password for data transfer [データ転送パスワード]
 * @return	Operation error code [動作エラーコード]
 */
long __stdcall DT2000_Download(const wchar_t* package_filename, const wchar_t* got_drive_name, long init_sram, long notreboot, 
	const wchar_t* project_username, const wchar_t* project_password, 
	const wchar_t* remote_password, const wchar_t* got_password
);


/** 
 * @brief	DT2000_ResourceDown()
 * @note	Write the resource file. [リソースファイルを書き込む。]
 *			Regard an empty text as being specified if NULL is specified for an argument. [引数にNULLを指定した場合、空の文字列が指定されたと見なす。]
 *			Display an error prior to transfer if the number of characters set for a password exceeds its maximum. [パスワードが最大文字数より大きい場合、転送前にエラーとする。]
 * @param	filenames		Resource file to be written [書き込むリソースファイル]
 * @param	got_drive_name	GOT drive [GOTドライブ]
 * @param	down_folder		Write destination folder[書込先フォルダ]
 * @param	remote_password	Remote password [リモートパスワード]
 * @param	got_password	Password for data transfer [データ転送パスワード]
 * @return	Operation error code [動作エラーコード]
 */
long __stdcall DT2000_ResourceDown(const wchar_t* filenames, const wchar_t* got_drive_name, const wchar_t* down_folder, 
	const wchar_t* remote_password, const wchar_t* got_password
);


/** 
 * @brief	DT2000_Upload()
 * @note	Retrieve the project file. [プロジェクトファイルを読み出す。]
 *			Regard an empty text as being specified if NULL is specified for an argument. [引数にNULLを指定した場合、空の文字列が指定されたと見なす。]
 *			Display an error prior to transfer if the number of characters set for a password exceeds its maximum. [パスワードが最大文字数より大きい場合、転送前にエラーとする。]
 * @param	package_filename	Readout package [読出パッケージ]
 * @param	remote_password		Remote password [リモートパスワード]
 * @param	got_password		Password for data transfer [データ転送パスワード]
 * @return	Operation error code [動作エラーコード]
 */
long __stdcall DT2000_Upload(const wchar_t* package_filename, const wchar_t* got_drive_name, 
	const wchar_t* remote_password, const wchar_t* got_password
);


/** 
 * @brief	DT2000_ResourceUp()
 * @note	Retrieve the resource data. [リソースデータを読み出す。]
 *			Regard an empty text as being specified if NULL is specified for an argument. [引数にNULLを指定した場合、空の文字列が指定されたと見なす。]
 * @param	up_folder		Readout destination folder[読出先フォルダ]
 * @param	filenames		Resource file to be retrieved [読み出すリソースファイル]
 * @param	got_drive_name	GOT drive [GOTドライブ]
 * @param	remote_password	Remote password [リモートパスワード]
 * @param	got_password	Password for data transfer [データ転送パスワード]
 * @return	Operation error code [動作エラーコード]
 */
long __stdcall DT2000_ResourceUp(const wchar_t* up_folder, const wchar_t* filenames, const wchar_t* got_drive_name, 
	const wchar_t* remote_password, const wchar_t* got_password
);


/** 
 * @brief	DT2000_CommConfig()
 * @note 	Change the communication setting. [通信設定を変更する。]
 *			Setting in which an empty text, NULL being specified should not be changed. [空文字列、NULLが指定された設定は変更しない。]
 *			Specification of IP address such as .01. is unacceptable. [IPアドレスは.01.のような指定は不可。]
 *			Up to 5 digits can be specified for port No. [ポート番号は最大5桁までで指定。]
 *			Display an error if the communication path becomes invalid after its change. [変更後無効な通信経路となる場合は、エラーとなる。]
 * @param	type			Connection method (USB, ETHERNET) [接続方法(USB, ETHERNET)]
 * @param	cport			Not support (ignore the specification) [未対応(指定は無視する)]
 * @param	baudrate		Not support (ignore the specification) [未対応(指定は無視する)]
 * @param	got_ip			GOT IP address [GOT IPアドレス]
 * @param	got_port		GOT IP port No. (1024-65534) (GOT IPのポート番号(1024～65534)]
 * @param	timeout_direct	Timeout (1-9999) [タイムアウト(1～9999)]
 * @param	retry_direct	Retry (0-5) [リトライ回数(0～5)]
 * @return	Operation error code (動作エラーコード)
 */
long __stdcall DT2000_CommConfig(const wchar_t* type, 
	const wchar_t* cport, const wchar_t* baudrate, 
	const wchar_t* got_ip, const wchar_t* got_port, 
	const wchar_t* timeout_direct, const wchar_t* retry_direct
);

/** 
 * @brief	DT2000_CommConfigPlc()
 * @note 	Change the communication setting.(Via PLC, PLC side I/F:PLC) [通信設定を変更する。(シーケンサ経由、シーケンサ側I/F:PLC)]
 *			Setting in which an empty text, NULL being specified should not be changed. [空文字列、NULLが指定された設定は変更しない。]
 *			Specification of IP address such as .01. is unacceptable. [IPアドレスは.01.のような指定は不可。]
 *			Up to 5 digits can be specified for port No. [ポート番号は最大5桁までで指定。]
 *			Display an error if the communication path becomes invalid after its change. [変更後無効な通信経路となる場合は、エラーとなる。]
 * @param	type				Connection method (USB, ETHERNET, RS232) [接続方法(USB, ETHERNET, RS232)]
 * @param	plc					PLC side I/F (RCPU, QCPU, LCPU) [シーケンサ側I/F(RCPU, QCPU, LCPU)]
 * @param	cport				Communication port (COM1-COM63) [通信ポート(COM1～COM63)]
 * @param	baudrate			Baudrate (115200, 57600, 38400, 19200, 9600) [ボーレート(115200, 57600, 38400, 19200, 9600)]
 * @param	pc_protocol			Protocol (TCP, UDP) [プロトコル(TCP, UDP)]
 * @param	plc_connect_type	PLC Connection method (EthernetPort, ViaHub) [シーケンサ接続時の接続方法(EthernetPort, ViaHub)]
 * @param	plc_connect_format	Specification method of destination for PLC connection (IP, HostName) [シーケンサ接続時の接続先の指定方法(IP, HostName)]
 * @param	plc_ip				PLC IP address [シーケンサIPアドレス]
 * @param	plc_host			PLC Host name [シーケンサホスト名]
 * @param	got_network			GOT network No. (1-239) [GOTのネットワークNo.(1～239)]
 * @param	got_station			GOT station No. (1-120) [GOTの局番(1～120)]
 * @param	timeout				Timeout (1-9999) [タイムアウト(1～9999)]
 * @param	retry				Retry (0-5) [リトライ回数(0～5)]

 * @return	Operation error code (動作エラーコード)
 */
long __stdcall DT2000_CommConfigPlc(
	const wchar_t* type, const wchar_t* plc, 
	const wchar_t* cport, const wchar_t* baudrate, 
	const wchar_t* pc_protocol, 
	const wchar_t* plc_connect_type, const wchar_t* plc_connect_format, const wchar_t* plc_ip, const wchar_t* plc_host, 
	const wchar_t* got_network, const wchar_t* got_station, 
	const wchar_t* timeout, const wchar_t* retry
);

/** 
 * @brief	DT2000_CommConfigEthernetUnit()
 * @note 	Change the communication setting.(Via PLC, PLC side I/F:EthernetUnit) [通信設定を変更する。(シーケンサ経由、シーケンサ側I/F:イーサネットユニット)]
 *			Setting in which an empty text, NULL being specified should not be changed. [空文字列、NULLが指定された設定は変更しない。]
 *			Specification of IP address such as .01. is unacceptable. [IPアドレスは.01.のような指定は不可。]
 *			Up to 5 digits can be specified for port No. [ポート番号は最大5桁までで指定。]
 *			Display an error if the communication path becomes invalid after its change. [変更後無効な通信経路となる場合は、エラーとなる。]
 * @param	plc					PLC side I/F (RJ71EN71, QJ71E71, LJ71E71) [シーケンサ側I/F(RJ71EN71, QJ71E71, LJ71E71)]
 * @param	pc_protocol			Protocol (TCP, UDP) [プロトコル(TCP, UDP)]
 * @param	pc_unit_network		PC network No. (1-239) [PCのネットワークNo.(1～239)]
 * @param	pc_unit_station		PC station No. (1-120) [PCの局番(1～120)]
 * @param	unit_station		Ethernet unit station No. (1-120) [Ethernetユニットの局番(1～120)]
 * @param	unit_connect_format	Specification method of destination for Ethernet unit connection (IP, HostName) [Ethernetユニット接続時の接続先の指定方法(IP, HostName)]
 * @param	unit_ip				Ethernet unit IP address [EthernetユニットIPアドレス]
 * @param	unit_host			Ethernet unit host name [Ethernetユニットホスト名]
 * @param	unit_staion_ip_info	Station No. <-> IP Info (AutoResponseSystem, Others) [局番<->IP関連情報(AutoResponseSystem, Others)]
 * @param	got_network			GOT network No. (1-239) [GOTのネットワークNo.(1～239)]
 * @param	got_station			GOT station No. (1-120) [GOTの局番(1～120)]
 * @param	timeout				Timeout (1-9999) [タイムアウト(1～9999)]
 * @param	retry				Retry (0-5) [リトライ回数(0～5)]

 * @return	Operation error code (動作エラーコード)
 */
long __stdcall DT2000_CommConfigEthernetUnit(
	const wchar_t* plc, 
	const wchar_t* pc_protocol, const wchar_t* pc_unit_network, const wchar_t* pc_unit_station, 
	const wchar_t* unit_station, const wchar_t* unit_connect_format, const wchar_t* unit_ip, const wchar_t* unit_host, 
	const wchar_t* unit_staion_ip_info, 
	const wchar_t* got_network, const wchar_t* got_station, 
	const wchar_t* timeout, const wchar_t* retry
);

/**
 * @brief	DT2000_CommTest()
 * @note 	Execute a communication test. [通信テストを実行する。]
 * @param	remote_password		Remote password [リモートパスワード]
 * @return	Operation error code [動作エラーコード]
 */
long __stdcall DT2000_CommTest(const wchar_t* remote_password);


/**
 * @brief	DT2000_GetLastCommError()
 * @note 	Acquire the code of a communication error that occurs in the communication executed at the last. [最後に実行した通信で発生した通信エラーのコードを取得する。]
 *			Initialize communication errors with 0 at the time of every communication execution. [通信エラーは、通信実行時毎に0で初期化される。]
 *			Hence, return 0 if no error occurs in the previous communication. [そのため、前回の通信でエラーが発生しなかった場合は0を返す。]
 *			(Set an error code in the case of a communication test as well.) [通信テストの場合もエラーコードは設定される。]
 * @param	none
 * @return	Communication error code [通信エラーコード]
 */
long __stdcall DT2000_GetLastCommError();


/**
 *  @brief		ConvertFile2000_Recipe()
 *  @note		Recipe Convert Function [レシピ変換関数]
 *  @param		p_OriginalConversionFile,	///Original conversion file (Absolute Path + File Name) [変換元のファイル名(絶対パス+ファイル名)]
 *  @param		p_AfterFileType				///Destination file type [変換先のファイル種類]
 *  @return
 *	CONVERT_COMPLETE
 *	ERROR2000_INCORRECTFILE
 *	ERROR2000_NO_FILE
 *	ERROR2000_NO_FILETYPE
 *	ERROR2000_NO_FILE_G2P
 */
int __stdcall ConvertFile2000_Recipe(
							wchar_t*	p_OriginalConversionFile,
							wchar_t*	p_AfterFileType
							);


/**
 *  @brief		ConvertFile2000_OPELOG()
 *  @note		Operation Log Conversion Function [操作ログ変換関数]
 *  @param		p_OriginalConversionFile,	///Original conversion file (Absolute Path + File Name) [変換元のファイル名(絶対パス+ファイル名)]
 *  @param		p_AfterFileType,			///Destination file type [変換先のファイル種類]
 *  @param		p_LanguageTypeAfterConvert	///Destination language type [変換先の言語タイプ]
 *  @return
 *	CONVERT_COMPLETE
 *	ERROR2000_INCORRECTFILE
 *	ERROR2000_NO_FILE
 *	ERROR2000_NO_FILETYPE
 *	ERROR2000_NO_LANGUAGETYPE
 *	ERROR2000_NO_FILE_OLCONVFILE
 */
int __stdcall ConvertFile2000_OPELOG(
							wchar_t*	p_OriginalConversionFile,
							wchar_t*	p_AfterFileType,
							wchar_t*	p_LanguageTypeAfterConvert
							);


/**
 *  @brief		ConvertFile2000_LOGGING()
 *  @note		Logging conversion function [ロギング変換関数]
 *  @param		p_OriginalConversionFile,	///Original conversion file (Absolute Path + File Name) [変換元のファイル名(絶対パス+ファイル名)]
 *  @param		p_AfterFileType				///Destination file type [変換先のファイル種類]
 *  @return
 *	CONVERT_COMPLETE
 *	ERROR2000_INCORRECTFILE
 *	ERROR2000_NO_FILE
 *	ERROR2000_NO_FILETYPE
 */
int __stdcall ConvertFile2000_LOGGING(
							wchar_t*	p_OriginalConversionFile,
							wchar_t*	p_AfterFileType
							);

/**
 *  @brief	 	ConvertFile2000_Alarm()
 *  @note		Alarm conversion function [アラーム変換関数]
 *  @param		p_OriginalConversionFile,	///Original conversion file (Absolute Path + File Name) [変換元のファイル名(絶対パス+ファイル名)]
 *  @param		p_ProjectFile,				///Project file (absolute path) to be used for conversion [変換に使用するプロジェクトファイル(絶対パス)]
 *  @param		p_AfterFileType,			///Destination file type [変換先のファイル種類]
 *  @param		p_LanguageTypeAfterConvert,	///Destination language type [変換先の言語タイプ]
 *  @param		p_ColumnNoAferConvert,		///Comment column No. (1-30) (User Alarm) [ユーザアラーム時のコメント列No.(1-30)]
 *  @param		p_TargetInSamePath,			///Target files in the same path (0, 1) [同じパス内のファイルを対象にするか(0,1)]
 *  @param		p_ProjectUser,				///Project user name [プロジェクトのユーザ名]
 *  @param		p_ProjectPassword			///Project password [プロジェクトのパスワード]
 *  @return
 *	CONVERT_COMPLETE
 *	ERROR2000_INCORRECTFILE
 *	ERROR2000_NO_FILE
 *	ERROR2000_NO_FILETYPE
 *	ERROR2000_NO_LANGUAGETYPE
 *	ERROR2000_NO_FILE_OLCONVFILE
 *	ERROR2000_NO_PROJECT
 *	ERROR2000_INCORRECT_PASSWORD
 *	ERROR2000_INCORRECT_PROJECT
 *	ERROR2000_CANNOT_CREATE_LOG
 *	ERROR2000_CANNOT_USE_PROJECT
 *	
 */
int __stdcall ConvertFile2000_Alarm(
							wchar_t*	p_OriginalConversionFile,
							wchar_t*	p_ProjectFile,
							wchar_t*	p_AfterFileType,
							wchar_t*	p_LanguageTypeAfterConvert,
							char		p_ColumnNoAferConvert,
							char		p_TargetInSamePath,
							wchar_t*	p_ProjectUser,
							wchar_t*	p_ProjectPassword
							);


#if !defined(_NATIVE_WCHAR_T_DEFINED) && !defined(DT_NOT_USE_US_DEFINE___)

#define DT2000_Download DT2000_DownloadUS
#define DT2000_ResourceDown DT2000_ResourceDownUS
#define DT2000_Upload DT2000_UploadUS
#define DT2000_ResourceUp DT2000_ResourceUpUS
#define DT2000_CommConfig DT2000_CommConfigUS
#define DT2000_CommConfigPlc DT2000_CommConfigPlcUS
#define DT2000_CommConfigEthernetUnit DT2000_CommConfigEthernetUnitUS
#define DT2000_CommTest DT2000_CommTestUS
#define DT2000_GetLastCommError DT2000_GetLastCommErrorUS

#define ConvertFile2000_Recipe ConvertFile2000_RecipeUS
#define ConvertFile2000_OPELOG ConvertFile2000_OPELOGUS
#define ConvertFile2000_LOGGING ConvertFile2000_LOGGINGUS
#define ConvertFile2000_Alarm ConvertFile2000_AlarmUS

#endif

long __stdcall DT2000_DownloadUS(const unsigned short* package_filename, const unsigned short* got_drive_name, long init_sram, long notreboot, 
	const unsigned short* project_username, const unsigned short* project_password, 
	const unsigned short* remote_password, const unsigned short* got_password
);
long __stdcall DT2000_ResourceDownUS(const unsigned short* filenames, const unsigned short* got_drive_name, const unsigned short* down_folder, 
	const unsigned short* remote_password, const unsigned short* got_password
);
long __stdcall DT2000_UploadUS(const unsigned short* package_filename, const unsigned short* got_drive_name, 
	const unsigned short* remote_password, const unsigned short* got_password
);
long __stdcall DT2000_ResourceUpUS(const unsigned short* up_folder, const unsigned short* filenames, const unsigned short* got_drive_name, 
	const unsigned short* remote_password, const unsigned short* got_password
);
long __stdcall DT2000_CommConfigUS(const unsigned short* type, 
	const unsigned short* cport, const unsigned short* baudrate, 
	const unsigned short* got_ip, const unsigned short* got_port, 
	const unsigned short* timeout_direct, const unsigned short* retry_direct
);
long __stdcall DT2000_CommConfigPlcUS(
	const unsigned short* type, const unsigned short* plc, 
	const unsigned short* cport, const unsigned short* baudrate, 
	const unsigned short* pc_protocol, 
	const unsigned short* plc_connect_type, const unsigned short* plc_connect_format, const unsigned short* plc_ip, const unsigned short* plc_host, 
	const unsigned short* got_network, const unsigned short* got_station, 
	const unsigned short* timeout, const unsigned short* retry
);
long __stdcall DT2000_CommConfigEthernetUnitUS(
	const unsigned short* plc, 
	const unsigned short* pc_protocol, const unsigned short* pc_unit_network, const unsigned short* pc_unit_station, 
	const unsigned short* unit_station, const unsigned short* unit_connect_format, const unsigned short* unit_ip, const unsigned short* unit_host, 
	const unsigned short* unit_staion_ip_info, 
	const unsigned short* got_network, const unsigned short* got_station, 
	const unsigned short* timeout, const unsigned short* retry
);
long __stdcall DT2000_CommTestUS(const unsigned short* remote_password);
long __stdcall DT2000_GetLastCommErrorUS();

int __stdcall ConvertFile2000_RecipeUS(
	unsigned short*	p_OriginalConversionFile,
	unsigned short*	p_AfterFileType
);
int __stdcall ConvertFile2000_OPELOGUS(
	unsigned short*	p_OriginalConversionFile,
	unsigned short*	p_AfterFileType,
	unsigned short*	p_LanguageTypeAfterConvert
);
int __stdcall ConvertFile2000_LOGGINGUS(
	unsigned short*	p_OriginalConversionFile,
	unsigned short*	p_AfterFileType
);
int __stdcall ConvertFile2000_AlarmUS(
	unsigned short*	p_OriginalConversionFile,
	unsigned short*	p_ProjectFile,
	unsigned short*	p_AfterFileType,
	unsigned short*	p_LanguageTypeAfterConvert,
	char		p_ColumnNoAferConvert,
	char		p_TargetInSamePath,
	unsigned short*	p_ProjectUser,
	unsigned short*	p_ProjectPassword
);

#endif //DT_FUNC2000_H___
