/****************************************************************************/
/* FILE NAME       : EntryWDTInt_R12.c                                      */
/* VERSION         : 1.00                                                   */
/* FUNCTION        : Register User WDT error interrupt                      */
/*                                                                          */
/*                                                                          */
/*  Create a task as given below and execute                                */
/*  1. EntryWDTMain()                                                       */
/*                                                                          */
/* When you want to build this sample,                                      */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder        */
/* in C Controller module to C:\WindRiver\INCLUDE.                          */
/*                                                                          */
/* Sample Program, use at own risk                                          */
/*                                                                          */
/* CREATED         : 2014/10/09   MITSUBISHI ELECTRIC CORPORATION           */
/*                                                                          */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED   */
/****************************************************************************/
#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <taskLib.h>
#include    <semLib.h>
#include    "CCPUFunc.h"

#define UNIT            2           /* Module identifier                    */
#define PRIORITY_150    150         /* Priority 150                         */
#define PRIORITY_160    160         /* Priority 160                         */
#define STACK           20000       /* Stack size 20000                     */
#define TASK_NAME_1     "SmpTask_1" /* Task name1                           */
#define TASK_NAME_2     "SmpTask_2" /* Task name2                           */

long    glWDTIntCnt = 0;            /* Total WDT interrupt count            */

/* Monitor Module Status                                                    */
void    StatusCheckRtn();
/* Processing to output the output signal(Y)                                */
void    YWord_WriteRtnTask();
/* WDT interrupt handler routine                                            */
void    WDTIntHookRtn();

/****************************************************************************/
/* MODULE   : EntryWDTMain                                                  */
/* ABSTRACT : WDT Main processing                                           */
/* FUNCTION : Task Initialization                                           */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-09  MITSUBISHI ELECTRIC CORPORATION                   */
/* UPDATE   :                                                               */
/****************************************************************************/
void EntryWDTMain()
{
    short   sRet;

    /************************************************************************/
    /* Create and activate a task monitoring Module Status                  */
    /************************************************************************/
    sRet = taskSpawn(TASK_NAME_1,
                    PRIORITY_150,
                    0,
                    STACK,
                    (FUNCPTR)StatusCheckRtn,
                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if(sRet == ERROR){
        printf("StatusCheckRtn task start error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }
    
    /************************************************************************/
    /* Create and activate a task processing to output the output signal(Y) */
    /************************************************************************/
    sRet = taskSpawn(TASK_NAME_2,
                    PRIORITY_160,
                    0,
                    STACK,
                    (FUNCPTR)YWord_WriteRtnTask,
                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if(sRet == ERROR){
        printf("YWord_WriteRtnTask task start error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    return;
}

/****************************************************************************/
/* MODULE   : StatusCheckRtn                                                */
/* ABSTRACT : Monitor Module Status                                         */
/* FUNCTION : WDT is controlled by monitoring the status of the unit        */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-09  MITSUBISHI ELECTRIC CORPORATION                   */
/* UPDATE   :                                                               */
/****************************************************************************/
void StatusCheckRtn()
{
    short          sRet;
    unsigned short pusErrInfo[3];
    long           lEvent[4];
    unsigned long  ulBufSize;

    /************************************************************************/
    /* Set timer event of 500ms cycle                                       */
    /************************************************************************/
    lEvent[0] = 1;    /* Timer event count                                  */
    lEvent[1] = 1;    /* Timer event No.1                                   */
    lEvent[2] = 500;  /* Timer event cycle(500ms)                           */
    lEvent[3] = 1;    /* Timer event synchronization type (1:individual)    */

    /************************************************************************/
    /* Register timer event of 500ms cycle.                                 */
    /************************************************************************/
    sRet = CCPU_EntryTimerEvent(lEvent);
    if(sRet != 0){
        printf("CCPU_EntryTimerEvent Error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /* Register a handler routine for the user WDT error interrupt          */
    /************************************************************************/
    sRet = CCPU_EntryWDTInt(0, WDTIntHookRtn);
    if(sRet != 0){
        printf("CCPU_EntryWDTInt Error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /* Start the user WDT (interval 1000ms = 1 second)                      */
    /************************************************************************/
    sRet = CCPU_StartWDT(0, 100);
    if(sRet != 0){
        printf("CCPU_StartWDT Error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /*  Every event of timer No.1 (500ms cycle interval) occurs,            */
    /*  do the following,                                                   */
    /*                                                                      */
    /*  1)Read the status information of the C Controller module            */
    /*  2)If an error occurred in the C Controller module, stop the         */
    /*    user WDT and then exit the process                                */
    /*  3)If no error has occured in the C Controller module,               */
    /*    reset the user WDT                                                */
    /************************************************************************/
    while(1){
        /********************************************************************/
        /*  Read the status information of the C Controller module          */
        /********************************************************************/
        ulBufSize = sizeof(pusErrInfo) / 2;
        sRet = CCPU_GetErrInfo(pusErrInfo, ulBufSize);
        if(sRet != 0){
            printf("\nCCPU_GetErrInfo Error  sRet = %d ( %hxH )\n", sRet, sRet);
            break;
        }

        /********************************************************************/
        /* Confirm error status of the C Controller module.                 */
        /* (Please customize this processing as per requirement)            */
        /********************************************************************/
        
        /********************************************************************/
        /* Check and Confirm if C Controller module has an error            */
        /********************************************************************/
        if(pusErrInfo[1] != 0 || pusErrInfo[2] != 0){   /* Error            */
            printf("Module Error Occurred!!  pusErrInfo[1] = 0x%x, pusErrInfo[2] = 0x%x\n", pusErrInfo[1], pusErrInfo[2]);
            /****************************************************************/
            /* Stop the user WDT                                            */
            /****************************************************************/
            sRet = CCPU_StopWDT(0);
            if(sRet != 0){
                printf("CCPU_StopWDT Error sRet = %d ( %hxH )\n", sRet, sRet);
            }
            break;
        }

        /********************************************************************/
        /* Reset the user WDT                                               */
        /********************************************************************/
        sRet = CCPU_ResetWDT(0);
        if(sRet != 0){
            printf("CCPU_ResetWDT Error sRet = %hd ( %hxH )\n", sRet, sRet);
            break;
        }

        /********************************************************************/
        /* Wait for the timer event No.1                    */
        /********************************************************************/
        sRet = CCPU_WaitTimerEvent(1);
        if(sRet != 0){
            printf("CCPU_WaitTimerEvent Error ret = %d ( %hxH )\n", sRet, sRet);
            break;
        }
    }

    return;
}

/****************************************************************************/
/* MODULE   : YWord_WriteRtnTask                                            */
/* ABSTRACT : Processing to output the output signal(Y)                     */
/* FUNCTION : Increment Bit Y10-Y1F every second                            */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-09  MITSUBISHI ELECTRIC CORPORATION                   */
/* UPDATE   :                                                               */
/****************************************************************************/
void    YWord_WriteRtnTask()
{
    short          sRet          = -1;
    unsigned short pusDataBuf[1] = {0};

    /************************************************************************/
    /* Increment Bit Y10 - Y1F every one second                             */
    /************************************************************************/
    pusDataBuf[0] = 0;
    
    while(1){
        /********************************************************************/
        /* Write Y10 - Y1F                                                  */
        /********************************************************************/

        sRet = CCPU_Y_Out_WordEx(0, 0x10, 1, pusDataBuf, 0);
        if(sRet != 0){
            printf("CCPU_Y_Out_WordEx Error sRet = %hd ( %hxH )\n", sRet, sRet);
            break;
        }

        /********************************************************************/
        /* Increment output data                                            */
        /********************************************************************/    
        pusDataBuf[0]++;
        
        /********************************************************************/
        /*  Read the status information of the C Controller module          */
        /*  using CCPU_GetCpuStatus() as per requirement and need           */
        /********************************************************************/
        /*                  User customized processing                      */

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    return;
}

/****************************************************************************/
/* MODULE   : WDTIntHookRtn                                                 */
/* ABSTRACT : WDT interrupt handler routine                                 */
/* FUNCTION : Register the WDT error to the event history                   */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-09  MITSUBISHI ELECTRIC CORPORATION                   */
/* UPDATE   :                                                               */
/****************************************************************************/
void    WDTIntHookRtn()
{
    short          sRet;
    unsigned short pusDataBufWR[1] = {0};
    unsigned short pusDataBufRD[1] = {0};

    /************************************************************************/
    /* Increment the WDT interrupt count                                    */
    /************************************************************************/
    glWDTIntCnt++;
    pusDataBufWR[0] = glWDTIntCnt;

    /************************************************************************/
    /* Write data to the battery backup RAM (User Area)                     */
    /************************************************************************/

    sRet = CCPU_WriteDevice_ISR(Dev_CCPU_ZR, 0, 1, pusDataBufWR);
    if(sRet != 0){
        return;
    }

    /************************************************************************/
    /* Read data from the battery backup RAM (User Area)                    */
    /************************************************************************/

    sRet = CCPU_ReadDevice_ISR(Dev_CCPU_ZR, 0, 1, pusDataBufRD);
    if(sRet != 0){
        return;
    }

    /************************************************************************/
    /* Register the event in the event history                              */
    /************************************************************************/

    sRet = CCPU_RegistEventLog(0x1111AAFF, "WDTIntHook from WDTIntHookRtn()");
    /* 0x1111AAFF : Code to register in the event history                   */
    if(sRet != 0){
        return;
    }

    return;
}
