/****************************************************************************/
/* FILE NAME       DevAccessChanCCIEC_R12.c                                */
/* VERSION         1.00                                                    */
/* FUNCTION        MELSEC Data Link Function (MD Function)                 */
/*                 Sample Program(CC-Link IE) for R12CCPU-V                */
/*                                                                         */
/*  Device Access via CC-Link IE Controller Network                        */
/*                                                                         */
/*  Content                                                                */
/*  - CCIEC_OwnAccessRtnMain                                               */
/*    When device LB20 (CC-Link IE Controller Network own station) is      */
/*    ON, read device LB30 to LB3F (own station) and write the read        */
/*    data to device LW0 (own station).                                    */
/*                                                                         */
/*  - CCIEC_OtherAccessRtnMain                                             */
/*    Read device LB20 to LB2F of CC-Link IE Controller Network other      */
/*    station No.2 and write the read data to device D0 and W20.           */
/*                                                                         */
/*    Channel 151                                                          */
/*    Station 1     Station 2                                              */
/*    Head I/O:0x00 Head I/O:0x00                                          */
/*  ______|___    ____ |______                                             */
/*  | C  |C |      |R |C |  |                                              */
/*  | C  |C |      |C |C |  |                                              */
/*  | P  |I |      |P |I |  |                                              */
/*  | U  |E |      |U |E |  |                                              */
/*  |    |C |      |  |C |  |                                              */
/*  |    |  |      |  |  |  |                                              */
/*  |____|__|__    |__|__|__|__                                            */
/*        |____________|                                                   */
/*          ChannelNo.1  Channel 151                                       */
/*                                                                         */
/*  Network Range Assignment                                               */
/*      ________________________________                                   */
/*      |______ |_LB_______|_LW________|                                   */
/*      |       |          |           |                                   */
/*      | STA 1 |  0-1F    |  0-1F     |                                   */
/*      |_______|__________|___________|                                   */
/*      |       |          |           |                                   */
/*      | STA 2 | 20-3F    | 20-3F     |                                   */
/*      |_______|__________|___________|                                   */
/*                                                                         */
/*  NOTE: For Link Device                                                  */
/*  CCPU        CCIEC               CCIEC       RCPU                       */
/*              STA 1               STA 2                                  */
/*_____________    _LB/LW___           _LB/LW___   _B/W____                */
/*|            |   |       |           |       |   |       |               */
/*|            |<=>| 0-1F -------------> 0-1F  |<=>|       |               */
/*|mdSendEx/   |   |_______|           |_______|   |_______|               */
/*|mdReceiveEx |   |       |           |       |   |       |               */
/*|            |<=>|20-3F <-------------20-3F  |<=>|       |               */
/*|____________|___|_______|__         |_______|___|_______|_              */
/*                                                                         */
/*  Link devices that are configured in the network range assignment is    */
/*  the value of the assigned device in code will always be reflected      */
/*  in the other stations.                                                 */
/*  Operation of CCIEC_OwnAccessRtnMain() signifies the above context      */
/*                                                                         */
/*  - Refer example for device value change by CCIEC_OwnAccessRtnMain()    */
/*  1)When LB20 of station No.2 is ON, LB20 of station No.1 is ON          */
/*                                                                         */
/*  2)Device LB20 of own station (station No.1) is ON, sample program      */
/*    reads the LB30 to LB3F of own station (station No.1)                 */
/*    (LB30 to LB3F of own station always reflect to the same device       */
/*    value of station No.2)                                               */
/*                                                                         */
/*  3)Read device LB30 to LB3F (own station) and write the read data       */
/*    to device LW0 of own station (station No.1).                         */
/*                                                                         */
/*  4)Since the value of device LW0 assigned to station No.1 changed,      */
/*    the value of LW0 of station No.2 changes so that the value are       */
/*    reflected by the other station                                       */
/*                                                                         */
/*  Note)                                                                  */
/*  When 3) writes the read data to device LW0 of other station            */
/*  (station No.2) instead of own station, the value of device LW0 does    */
/*   not change (so that a value of LW0 of station No.1 is reflected)      */
/*                                                                         */
/* When you want to build this sample,                                     */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder       */
/* in C Controller module to C:\WindRiver\INCLUDE.                         */
/*                                                                         */
/*  Sample Program, user at own risk                                       */
/*                                                                         */
/* CREATED         2014/10/22   MITSUBISHI ELECTRIC CORPORATION            */
/*                                                                         */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED  */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <usrLib.h>
#include    <taskLib.h>
#include    "MDFunc.h"
#include    "CCPUFunc.h"

#define OWN_STATION     0xFF    /* Station Number(Own Station)             */

#if 0
/*  If Link Refresh Access is required or need to use, change #if 0 -> 1    */
/*  Refresh parameter setting for device access                             */
#define __LINK_REFRESH_ON__     /* Link Refresh Access                      */
#endif

/* CC-Link IE Controller Network own station access processing              */
void CCIEC_OwnAccessRtn();
/* CC-Link IE Controller Network other station access processing            */
void CCIEC_OtherAccessRtn();

/****************************************************************************/
/* MODULE   : CCIEC_OwnAccessRtnMain                                        */
/* ABSTRACT : CC-Link IE Controller Network own station access processing   */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEC_OwnAccessRtnMain()
{
    /************************************************************************/
    /* Create Task(Use sp or taskSpawn)                                     */
    /************************************************************************/
    sp((FUNCPTR)CCIEC_OwnAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CCIEC_OwnAccessRtn                                            */
/* ABSTRACT : CC-Link IE Controller Network own station access              */
/* FUNCTION : Own station access                                            */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEC_OwnAccessRtn()
{
    long    lRet;
    long    lPath;
    short   sData;
    long    lSize;

    /************************************************************************/
    /* Open CC-Link IE Controller Network Unit1                             */
    /************************************************************************/
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access(Bus Access: Own Station) */
    lRet = mdOpen( 12, -1, &lPath ); /* In order to access it through the link refresh device, to open the R series bus interface (channel No.12)  */
#else
    lRet = mdOpen(151, -1, &lPath); /* In order to access it through the link device, to open the CC-Link IE Controller network unit (channel No.151)   */
#endif
    if(lRet != 0){
        printf("\nmdOpen Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
        return;
    }

    /************************************************************************/
    /* When device LB20 (own station) is ON, read device LB30 to LB3F       */
    /* (own station) and write the read data to device LW0 (own station)    */
    /************************************************************************/
    while(1){ /* Infinite Loop                                              */
        /********************************************************************/
        /* Read own Station device LB20                                     */
        /********************************************************************/
        lSize = 2;
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access */
        lRet = mdReceiveEx(lPath, 0, OWN_STATION, DevB, 0x20, &lSize, &sData);
#else                           /* Direct Access   */
        lRet = CCPU_ReadLinkDevice(0x0, Dev_LB, 0x20, lSize, (unsigned short*)&sData, lSize);
#endif
        if(lRet != 0){
            printf("\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
            mdClose(lPath);
            return;
        }

        /********************************************************************/
        /* Read and Write                                                   */
        /********************************************************************/
        if((sData & 0x0001) == 1){ /*Check if own Station device LB20 is ON */
            /****************************************************************/
            /* Read device LB30-LB3F (own station)                          */
            /****************************************************************/
            lSize = 2;
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access         */
            lRet = mdReceiveEx(lPath, 0, OWN_STATION, DevB, 0x30, &lSize, &sData);
#else                           /* Direct Access               */
            lRet = CCPU_ReadLinkDevice(0x0, Dev_LB, 0x30, lSize, (unsigned short*)&sData, lSize);
#endif
            if(lRet != 0){
                printf("\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
                mdClose(lPath);
                return;
            }

            /****************************************************************/
            /* Write the Read data to own station device LW0                */
            /****************************************************************/
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access         */
            lRet = mdSendEx(lPath, 0, OWN_STATION, DevW, 0, &lSize, &sData);
#else                           /* Direct Access               */
            lRet = CCPU_WriteLinkDevice(0x0, Dev_LW, 0, lSize, (unsigned short*)&sData, 0);
#endif
            if(lRet != 0){
                printf("\nmdSendEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
                mdClose(lPath);
                return;
            }
            break;
        }

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    /************************************************************************/
    /* Print the read data of device LB30-LB3F                             */
    /************************************************************************/
    printf("Read Data  (LB30-LB3F) = %hd ( %hxH )\n", sData, sData);

    /************************************************************************/
    /* Close CC-Link IE Controller Network Unit1                            */
    /************************************************************************/
    mdClose(lPath);
    return; 
}


/****************************************************************************/
/* MODULE   : CCIEC_OtherAccessRtnMain                                      */
/* ABSTRACT : CC-Link IE Controller Network other station access task       */
/* FUNCTION : Activate Task                                                 */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEC_OtherAccessRtnMain()
{
    /************************************************************************/
    /* Create Task (Use sp or taskSpawn)                                    */
    /************************************************************************/
    sp((FUNCPTR)CCIEC_OtherAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CCIEC_OtherAccessRtn                                          */
/* ABSTRACT : CC-Link IE Controller Network other station access            */
/* FUNCTION : Other Station Access                                          */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEC_OtherAccessRtn()
{
    long    lRet;
    long    lPath;
    long    lStationNo = 2;   /* Station No.2                               */
    short   sData;
    long    lSize;
    long    lNetNo = 1;       /* Network No.= 1                             */
    
    /************************************************************************/
    /* Open CC-Link IE Controller Network Unit 1(Channel No.151)            */
    /************************************************************************/
    lRet = mdOpen(151, -1, &lPath);    
    if(lRet != 0){
        printf("\nmdOpen Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
        return;
    }

    /************************************************************************/
    /* Read device LB20-LB2F of Station no.2                                */
    /************************************************************************/
    lSize = 2;
    lRet = mdReceiveEx(lPath, lNetNo, lStationNo, DevLB(lNetNo), 0x20, &lSize, &sData);
    if(lRet != 0){
        printf("\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Write the read data to device D0 of station No.2 (control CPU)       */
    /************************************************************************/
    lRet = mdSendEx(lPath, lNetNo, lStationNo, DevD, 0, &lSize, &sData);
    if(lRet != 0){
        printf("\nmdSendEx(D) Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Write the read data to device W20 of station No.2 (control CPU)      */
    /************************************************************************/
    lRet = mdSendEx(lPath, lNetNo, lStationNo, DevW, 0x20, &lSize, &sData);
    if(lRet != 0){
        printf("\nmdSendEx(W)Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Print the read data of device LB20-LB2F                              */
    /************************************************************************/
    printf("Read Data (LB20-LB2F) = %hd ( %hxH )\n", sData, sData);

    /************************************************************************/
    /* Close CC-Link IE Controller Network Unit1(Channel No.151)            */
    /************************************************************************/
    mdClose(lPath);
    return;
}
