/****************************************************************************/
/* FILE NAME       DevAccessChanCCIEF_R12.c                                */
/* VERSION         1.00                                                    */
/* FUNCTION        MELSEC Data Link Function (MD Function)                 */
/*                 Device Access via CC-Link IE Field Network              */
/*                 Sample Program for R12CCPU-V                            */
/*                                                                         */
/*  Device Access via CC-Link IE Field Network                             */
/*                                                                         */
/*  Contents                                                               */
/*  - CCIEF_OwnAccessRtnMain                                               */
/*    When device RX0(CC-Link IE Filed Network(Channel No.181, Network     */
/*    No.1) own station is ON, read device RX10 to RX1F of own station     */
/*    and write the read data to device RY10 to RY1F (own station)         */
/*                                                                         */
/*  - CCIEF_OtherAccessRtnMain                                             */
/*    Read device RX0-RXF of CC-Link IE Field Network (Channel No.181,     */
/*    Network No.1) own station and write the read data to device D0       */
/*    and RY0 to RYF                                                       */
/*                                                                         */
/*   Channel 181                                                           */
/*   Station 0     Station 1                                               */
/*   Head I/O:0x00 Head I/O:0x00                                           */
/*  ______|___    ____ |______                                             */
/*  | C  |C |      |R |C |  |                                              */
/*  | C  |C |      |C |C |  |                                              */
/*  | P  |I |      |P |I |  |                                              */
/*  | U  |E |      |U |E |  |                                              */
/*  |    |F |      |  |F |  |                                              */
/*  |    |  |      |  |  |  |                                              */
/*  |____|__|__    |__|__|__|__                                            */
/*        |____________|                                                   */
/*          Network No.1  Channel 181                                      */
/*                                                                         */
/*                                                                         */
/*  Note:For Link Device                                                   */
/*       The range of device RX and RY of CC-Link IE Field Network         */
/*       station no.1 is assigned to 0-7F                                  */
/*       The significance and relation of RX and RY in the above           */
/*       configuration is shown below                                      */
/*                                                                         */
/*  CCPU        CC-Link             CC-Link     RCPU                       */
/*              IE Field            IE Field                               */
/*              Master              Local                                  */
/*              station 0           Station 1                              */
/*  ____________    _RX______           _RY______   _(Device)_             */
/*  |           |   |       |           |       |   |       |              */
/*  |           |<- | 0-7F <------------- 0-7F  |<- |       |              */
/*  |mdSendEx   |   |_______|           |_______|   |       |              */
/*  |mdReceiveEx|   |       |           |       |   |       |              */
/*  |           |   _RY_____|           _RX_____|   |       |              */
/*  |           |   |       |           |       |   |       |              */
/*  |           |-> | 0- 7F -------------> 0-7F |-> |       |              */
/*  |___________|___|_______|           |_______|___|_______|__            */
/*                                                                         */
/* When you want to build this sample,                                     */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder       */
/* in C Controller module to C:\WindRiver\INCLUDE.                         */
/*                                                                         */
/*  Sample Program, use at own risk                                        */
/*                                                                         */
/* CREATED         :2014/10/22   MITSUBISHI ELECTRIC CORPORATION           */
/*                                                                         */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED  */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <usrLib.h>
#include    "MDFunc.h"

#define OWN_STATION     0xFF    /* Station No.(Own Station)                 */

#if 0
/*  If Link Refresh Access is required or need to use, change #if 0 -> 1    */
/*  - Refresh parameter setting for device access                            */
#define __LINK_REFRESH_ON__     /* Link Refresh Access                      */
#endif

/* CC-Link IE Field Network own station access processing                   */
void CCIEF_OwnAccessRtn();
/* CC-Link IE Field Network other station access processing                 */
void CCIEF_OtherAccessRtn();

/****************************************************************************/
/* MODULE   : CCIEF_OwnAccessRtnMain                                        */
/* ABSTRACT : CC-Link IE Field Network Own station access task              */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEF_OwnAccessRtnMain()
{
    /************************************************************************/
    /* Create Task(Use sp or taskSpawn)                                     */
    /************************************************************************/
    sp((FUNCPTR)CCIEF_OwnAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CCIEF_OwnAccessRtn                                            */
/* ABSTRACT : CC-Link IE Field Network own station access                   */
/* FUNCTION : Own Station access                                            */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEF_OwnAccessRtn()
{
    long    lRet;
    long    lPath;
    short   sData;
    long    lSize;

    /************************************************************************/
    /* Open CC-Link IE Field Network Unit1                                  */
    /************************************************************************/
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access(Bus Access: Own Station) */
    lRet = mdOpen( 12, -1, &lPath );    /* In order to access through link refresh device, to open the R series bus interface (channel No.12)  */
#else
    lRet = mdOpen( 12, -1, &lPath );    /* In order to access it through the Intelligent function module buffer memory, to open the R series bus interface (channel No.12)  */
#endif
    if ( lRet != 0 ){
        printf( "\nmdOpen Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
        return;
    }

    /************************************************************************/
    /* When own station device RX0 is ON, read device RX10 to RX1F of     */
    /* own station and write the read data to device RY10 to RY1F of      */
    /* own station                                                        */
    /************************************************************************/
    while ( 1 ){ /* Infinite Loop                                           */
        /********************************************************************/
        /* Read device RX0 of own station                                   */
        /********************************************************************/
        lSize = 2;
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access(RX0->B0) */
        lRet = mdReceiveEx( lPath, 0, OWN_STATION, DevB, 0x0, &lSize, &sData );
#else                           /* Access through Intelligent Function Module Buffer Memory */
        lRet = mdReceiveEx( lPath, 0, OWN_STATION, DevSPG(0), 0x0, &lSize, &sData );
#endif
        if ( lRet != 0 ){
            printf( "\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
            mdClose( lPath );
            return;
        }

        /********************************************************************/
        /* Read and Write                                                   */
        /********************************************************************/
        if ( (sData & 0x0001) == 1 ){ /* Check if device RX0 is ON          */
            /****************************************************************/
            /* Read device RX10 to RX1F of own station                      */
            /****************************************************************/
            lSize = 2;
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access (RX10->B10)                           */
            lRet = mdReceiveEx( lPath, 0, OWN_STATION, DevB, 0x10, &lSize, &sData );
#else                           /* Access through Intelligent Function Module Buffer Memory */
            lRet = mdReceiveEx( lPath, 0, OWN_STATION, DevSPG(0), 0x1, &lSize, &sData );
#endif
            if ( lRet != 0 ){
                printf( "\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
                mdClose( lPath );
                return;
            }

            /****************************************************************/
            /* Write the read data to device RY10 to RY1F of own station    */
            /****************************************************************/
#ifdef __LINK_REFRESH_ON__      /* Link Refresh Access(RY10<-B90)                            */
            lRet = mdSendEx( lPath, 0, OWN_STATION, DevB, 0x90, &lSize, &sData );
#else                           /* Access through Intelligent Function Module Buffer Memory */
            lRet = mdSendEx( lPath, 0, OWN_STATION, DevSPG(0), 0x401, &lSize, &sData );
#endif
            if ( lRet != 0 ){
                printf( "\nmdSendEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
                mdClose(lPath);
                return;
            }
            break;
        }

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    /************************************************************************/
    /* Print the read data of device RX10 to RX1F                           */
    /************************************************************************/
    printf("Read Data  (RX10-RX1F) = %hd ( %hxH )\n", sData, sData);

    /************************************************************************/
    /* Close CC-Link IE Field Network Unit1                                 */
    /************************************************************************/
    mdClose(lPath);
    return; 
}

/****************************************************************************/
/* MODULE   : CCIEF_OtherAccessRtnMain                                      */
/* ABSTRACT : CC-Link IE Field Network other station access task            */
/* FUNCTION : start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEF_OtherAccessRtnMain()
{
    /************************************************************************/
    /* Create Task (Use sp or taskSpawn)                                    */
    /************************************************************************/
    sp( (FUNCPTR)CCIEF_OtherAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0) ;
    return;
}

/****************************************************************************/
/* MODULE   : CCIEF_OtherAccessRtn                                          */
/* ABSTRACT : CC-Link IE Field Network other station access                 */
/* FUNCTION : Other station access                                          */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CCIEF_OtherAccessRtn()
{
    long    lRet;
    long    lPath;
    long    lStationNo = 1;   /* Station no.1                               */
    short   sData;
    long    lSize;
    long    lNetNo = 1;       /* Network No.=1                              */

    /************************************************************************/
    /* Open CC-Link IE Field Network Unit 1(Channel No.181)                 */
    /************************************************************************/
    lRet = mdOpen( 181, -1, &lPath );
    if ( lRet != 0 ){
        printf( "\nmdOpen Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
        return;
    }

    /************************************************************************/
    /* Read device RX0 to RXF of station no.1                               */
    /************************************************************************/
    lSize = 2;
    lRet = mdReceiveEx( lPath, lNetNo, lStationNo, DevLX(lNetNo), 0x0, &lSize, &sData );
    if ( lRet != 0) {
        printf( "\nmdReceiveEx Error lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
        mdClose( lPath );
        return;
    }

    /************************************************************************/
    /* Write the read data to device D0 of station No.1 (Management CPU)    */
    /************************************************************************/
    lRet = mdSendEx( lPath, lNetNo, lStationNo, DevD, 0, &lSize, &sData );
    if ( lRet != 0 ){
        printf( "\nmdSendEx(D)Error  lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
        mdClose( lPath );
        return;
    }

    /*****************************************************************************/
    /* Write the read data to device RY0 to RYF of station No.1 (Management CPU) */
    /*****************************************************************************/
    lRet = mdSendEx( lPath, lNetNo, lStationNo, DevLY(lNetNo), 0x0, &lSize, &sData );
    if ( lRet != 0 ){
        printf( "\nmdSendEx(Y)Error  lRet = %hd ( %hxH )\n", (short)lRet, (short)lRet );
        mdClose( lPath );
        return;
    }

    /************************************************************************/
    /* Print the read data of device RX0 to RXF                             */
    /************************************************************************/
    printf( "Read Data (RX0-RXF) = %hd ( %hxH )\n", sData, sData );

    /************************************************************************/
    /* Close CC-Link IE Field Network Unit1(Channel No.181)                 */
    /************************************************************************/
    mdClose( lPath );
    return;
}
