/****************************************************************************/
/* FILE NAME       DevAccessChanCCL_R12.c                                   */
/* VERSION         1.00                                                     */
/* FUNCTION        MELSEC Data Link Function(MD Function)                   */
/*                 (Sample Program for R12CCPU-V)                           */
/*                                                                          */
/*  Device Access via CC-Link                                               */
/*                                                                          */
/*  Contents                                                                */
/*  - CC_OwnAccessRtnMain                                                   */
/*    When device RX0(CC-Link (Channel No.81)) own station                  */
/*    is ON, read device RX10 to RX1F of own station and write the read     */
/*    data to device RY10 to RY1F (own station)                             */
/*                                                                          */
/*  - CC_OtherAccessRtnMain                                                 */
/*    Read device RX0 to RXF of CC-Link (Channel No.81)                     */
/*    own station and write the read data to device D0 and RY0 to RYF       */
/*                                                                          */
/*                                                                          */
/*    Channel 81                                                            */
/*    Master        Local                                                   */
/*    Station       Station                                                 */
/*    STA.0         STA.1(4 exclusive)                                      */
/*    Head I/O:0x00 Head I/O:0x00                                           */
/*  ______|___    ____ |______                                              */
/*  | C  |C |      |R |C |  |                                               */
/*  | C  |C |      |C |C |  |                                               */
/*  | P  |L |      |P |L |  |                                               */
/*  | U  |i |      |U |i |  |                                               */
/*  |    |n |      |  |n |  |                                               */
/*  |    |k |      |  |k |  |                                               */
/*  |____|__|__    |__|__|__|__                                             */
/*        |____________|                                                    */
/*       Channel 81                                                         */
/*                                                                          */
/*  Note: For Link Device                                                   */
/*       The range of device RX and RY of CC-Link                           */
/*       station no.1 is assigned to 0-7F                                   */
/*       The significance and relation of RX and RY in the above            */
/*       configuration is shown below                                       */
/*                                                                          */
/*  CCPU        CC-Link             CC-Link     RCPU                        */
/*              Master              Local                                   */
/*              Station 0           Station 1(4 Occupied)                   */
/*  ____________    _RX______           _RY______   _(Device)_              */
/*  |           |   |       |           |       |   |       |               */
/*  |           |<- | 0-7F <------------- 0-7F  |<- |       |               */
/*  |mdSendEx/  |   |_______|           |_______|   |       |               */
/*  |mdReceiveEx|   |       |           |       |   |       |               */
/*  |           |   _RY_____|           _RX_____|   |       |               */
/*  |           |   |       |           |       |   |       |               */
/*  |           |-> | 0-7F --------------> 0-7F |-> |       |               */
/*  |___________|___|_______|           |_______|___|_______|__             */
/*                                                                          */
/*  *1  For the communication between Master station and local station,     */
/*      the last 2 bits (in this case 7E and 7F) can not be used.           */
/*                                                                          */
/* When you want to build this sample,                                      */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder        */
/* in C Controller module to C:\WindRiver\INCLUDE.                          */
/*                                                                          */
/*  Sample Program, user at own risk                                        */
/*                                                                          */
/* CREATED         : 2014/10/22   MITSUBISHI ELECTRIC CORPORATION           */
/*                                                                          */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED   */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <usrLib.h>
#include    <taskLib.h>
#include    "MDFunc.h"

#define OWN_STATION     0xFF    /* Station No.(Own Station)                 */

/* CC-Link own station access processing                                    */
void CC_OwnAccessRtn();
/* CC-Link other station access processing                                  */
void CC_OtherAccessRtn();

/****************************************************************************/
/* MODULE   : CC_OwnAccessRtnMain                                           */
/* ABSTRACT : CC-Link own station access task                               */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CC_OwnAccessRtnMain()
{
    /************************************************************************/
    /* Create Task (Use sp or taskSpawn)                                    */
    /************************************************************************/
    sp((FUNCPTR)CC_OwnAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CC_OwnAccessRtn                                               */
/* ABSTRACT : CC-Link own station access                                    */
/* FUNCTION : Own station access                                            */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CC_OwnAccessRtn()
{
    short   sRet;
    long    lPath;
    short   sData;
    long    lSize;

    /************************************************************************/
    /* Open CC-Link Unit1                                                   */
    /************************************************************************/
    sRet = mdOpen(12, -1, &lPath);    /* In order to access it through the Intelligent function module buffer memory, to open the R series bus interface (channel No.12)  */
    if(sRet != 0){
        printf("\nmdOpen Error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /* When own station device RX0 is ON, read device RX10 to RX1F of       */
    /* own station and write the read data to device RY10 to RY1F of        */
    /* own station                                                          */
    /************************************************************************/
    while(1){  /* Infinite Loop                                             */
        /********************************************************************/
        /* Read device RX0 of own station                                 */
        /********************************************************************/
        lSize = 2;   /* Access through Intelligent Function Module Buffer Memory */
        sRet = mdReceiveEx(lPath,0, OWN_STATION, DevSPG(0), 0xE0, &lSize, &sData);
        if(sRet != 0){
            printf("\nmdReceiveEx Error  sRet = %hd ( %hxH )\n", sRet, sRet);
            mdClose(lPath);
            return;
        }

        /********************************************************************/
        /* Read and Write                                                   */
        /********************************************************************/
        if((sData & 0x0001) == 1){  /* Check if device RX0 is ON            */
            /****************************************************************/
            /* Read device RX10 to RX1F of own station                      */
            /****************************************************************/
            lSize = 2;   /* Access through Intelligent Function Module Buffer Memory */
            sRet = mdReceiveEx(lPath,0, OWN_STATION, DevSPG(0), 0xE1, &lSize, &sData);
            if(sRet != 0){
                printf("\nmdReceiveEx Error  sRet = %hd ( %hxH )\n", sRet, sRet);
                mdClose(lPath);
                return;
            }

            /****************************************************************/
            /* Write the read data to device RY10 to RY1F of own station)   */
            /****************************************************************/
            /* Access through Intelligent Function Module Buffer Memory */
            sRet = mdSendEx(lPath, 0, OWN_STATION, DevSPG(0), 0x161, &lSize, &sData);
            if(sRet != 0){
                printf("\nmdSendEx Error  sRet = %hd ( %hxH )\n", sRet, sRet);
                mdClose(lPath);
                return;
            }
            break;
        }

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    /************************************************************************/
    /* Print the read data of device RX10 to RX1F                           */
    /************************************************************************/
    printf("Read Data  (RX10-RX1F) = %hd ( %hxH )\n", sData, sData);

    /************************************************************************/
    /* Close CC-Link Unit1                                                  */
    /************************************************************************/
    mdClose(lPath);
    return; 
}

/****************************************************************************/
/* MODULE   : CC_OtherAccessRtnMain                                         */
/* ABSTRACT : CC-Link other station access task                             */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CC_OtherAccessRtnMain()
{
    /************************************************************************/
    /* Create Task (Use sp or taskSpawn)                                    */
    /************************************************************************/
    sp((FUNCPTR)CC_OtherAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CC_OtherAccessRtn                                             */
/* ABSTRACT : CC-Link other station access                                  */
/* FUNCTION : Other Station Access                                          */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CC_OtherAccessRtn()
{
    short   sRet;
    long    lPath;
    short   sData;
    long    lSize;
    long    lStationNo = 1;   /* Station No.1                               */

    /************************************************************************/
    /* Open CC-Link Unit1 (Channel No.81)                                   */
    /************************************************************************/
    sRet = mdOpen(81, -1, &lPath);
    if(sRet != 0){
        printf("\nmdOpen Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /* Read device RX0 to RXF of Station No.1                               */
    /************************************************************************/
    lSize = 2;  /* Access through Intelligent Function Module Buffer Memory */
    sRet = mdReceiveEx(lPath, 0, lStationNo, DevSPG(0), 0xE0, &lSize, &sData);
    if(sRet != 0){
        printf("\nmdReceiveEx Error sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Write the read data to device D0 of Station No.1 (Management CPU)    */
    /************************************************************************/
    sRet = mdSendEx(lPath, 0, lStationNo, DevD, 0, &lSize, &sData);
    if(sRet != 0){
        printf("\nmdSendEx(D)Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Write the read data to device RY0 to RYF of Station No.1             */
    /************************************************************************/
#if 1       /* Access through Intelligent Function Module Buffer Memory when RY refresh device setting is not set */
    sRet = mdSendEx(lPath, 0, lStationNo, DevSPG(0), 0x160, &lSize, &sData);
#else       /* RY with refresh device set(RY<-M0)                   */
    sRet = mdSendEx(lPath, 0, lStationNo, DevM, 0, &lSize, &sData);
#endif
    if(sRet != 0){
        printf("\nmdSendEx(RY)Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Print the read data of device RX0 to RXF                             */
    /************************************************************************/
    printf("Read Data (RX0-RXF) = %hd ( %hxH )\n", sData, sData); 

    /************************************************************************/
    /* Close CC-Link Unit1(Channel No.81)                                   */
    /************************************************************************/
    mdClose(lPath);
    return;
}
