/********************************************************************************/
/* FILE NAME       : RD75Status_R12.c                                           */
/* VERSION         : 1.00                                                       */
/* FUNCTION        : It reads from the positioning unit and it monitors it the  */
/*                   one shot                                                   */
/*                   (For R12CCPU-V)                                            */
/*                                                                              */
/*  This is the sample program that reads the contents from following           */
/*  positioning unit, and executes the 1 shot monitoring                        */
/*  (it means communicate once).                                                */
/*                                                                              */
/*  - This program is supported with following positioning unit only            */
/*    Please note that other positioning unit cannot be used                    */
/*                                                                              */
/*  - Supported Positioning Units                                               */
/*  RD75P**                                                                     */
/*  RD75D**                                                                     */
/*                                                                              */
/*  - One-shot monitor content                                                  */
/*    (1) Current feed value                                                    */
/*    (2) Actual current value (RD75M or RD75MH only)                           */
/*    (3) Axis error No.                                                        */
/*    (4) Axis warning No.                                                      */
/*    (5) Status                                                                */
/*    (6) Servo status (RD75M or RD75MH only)                                   */
/*                                                                              */
/* When you want to build this sample,                                          */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder            */
/* in C Controller module to C:\WindRiver\INCLUDE.                              */
/*                                                                              */
/*  Sample Program, User at Own Risk                                            */
/*                                                                              */
/* CREATED         : 2014/10/22   MITSUBISHI ELECTRIC CORPORATION               */
/*                                                                              */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED       */
/********************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "CCPUFunc.h"

/********************************************************************************/
/* Offset address of axis No.1                                                  */
/********************************************************************************/
#define CURRENT_FEED_VAL    800 /* Md.20    Buffer memory of the current feed value */
#define AXSIS_ERR_NO        806 /* Md.23    Buffer memory of the axis error No.     */
#define AXSIS_WARNING_NO    807 /* Md.24    Buffer memory of the axis warning No.   */
#define STATUS              817 /* Md.31    Buffer memory of the status             */
#define REAL_CURRENT_VAL    850 /* Md.101   Buffer memory of the actual current No. */
#define SERVO_STATUS        877 /* Md.108   Buffer memory of the servo status       */

/********************************************************************************/
/* Offset size of 1-Axis (Next Axis)                                            */
/********************************************************************************/
#define AXIS_OFSET_SIZE     100

/********************************************************************************/
/* Model Name of Positioning Unit                                               */
/********************************************************************************/
#define UNITTYPE_RD75P        0 /* Model name of the positioning Unit RD75P     */
#define UNITTYPE_RD75D        1 /* Model name of the positioning Unit RD75D     */
#define UNITTYPE_RD75M        2 /* Model name of the positioning Unit RD75M     */
#define UNITTYPE_RD75MH       3 /* Model name of the positioning Unit RD75MH    */

/*  RD75 status monitoring routine                                              */
short RD75Status( unsigned short usIoNo, short sType);

/********************************************************************************/
/* MODULE   : RD75Status                                                        */
/* ABSTRACT : RD75 status monitoring routine                                    */
/* FUNCTION : One shot monitor of positioning unit                              */
/* NOTE     : Please specify the following for sType                            */
/*            positioning Unit name                                             */
/*            0:RD75P                                                           */
/*            1:RD75D                                                           */
/*            2:RD75M                                                           */
/*            3:RD75MH                                                          */
/* RETURN   :                                                                   */
/*            0             Normal                                              */
/*            Otherwise     Error                                               */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                     */
/* UPDATE   :                                                                   */
/********************************************************************************/
short RD75Status(
     unsigned short usIoNo,             /*  (I)Start I/O No.(Example:0x20)        */
     short sType)                       /*  (I)Model name of the positioning Unit */
{
    short               sRet;           /* Return value                           */
    unsigned long       ulOffset;       /* Offset of the buffer memory            */
    unsigned long       ulSize;         /* Read number of words                   */
    unsigned short      pusDataBuf[2];  /* Read data                              */
    unsigned long       ulBufSize;      /* Size of data storage area "pusDataBuf" */
    short               sLoop;          /* Variable for loop                      */

    /****************************************************************************/
    /* Read Data of 4 Axis                                                      */
    /****************************************************************************/
    long                plCurrentFeedVal[4]; /* Current feed value              */
    long                plRealCurrentVal[4]; /* Actual current value            */
    unsigned short      pusAxisErrNo[4];     /* Error No.                       */
    unsigned short      pusAxisWarningNo[4]; /* Warning No.                     */
    unsigned short      pusStatus[4];        /* Status                          */
    unsigned short      pusServoStatus[4];   /* Servo status                    */
    /****************************************************************************/
    /* Check the range of the model name of the positioning Unit                */
    /****************************************************************************/
    if( (sType < UNITTYPE_RD75P) || (sType > UNITTYPE_RD75MH) ){
        printf("RD75 Unit type error\n");
        return 1;
    }

    /****************************************************************************/
    /* Md.20 Read the current feed value                                        */
    /****************************************************************************/
    ulOffset = CURRENT_FEED_VAL;        /* Offset address of the current feed value  */
    ulSize = 2;                         /* Read word size                            */
    ulBufSize = 2;                      /* Read buffer size                          */
    for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis                   */
        sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
        
        if(sRet != 0){                  /*  In case of error                         */
            printf("CCPU_FromBuf Error Md.20 [ret = %hd]\n", sRet);
            return sRet;
        }
        plCurrentFeedVal[sLoop] = (long)(pusDataBuf[0] + (pusDataBuf[1] << 16)); /* copy current feed value to variable */
        ulOffset += AXIS_OFSET_SIZE;                                             /* Add the offset size per axis        */
    }

    /****************************************************************************/
    /* Md.23 Read the axis error No.                                            */
    /****************************************************************************/
    ulOffset = AXSIS_ERR_NO;            /* Offset address of the axis error No. */
    ulSize = 1;                         /* Read word size                       */
    ulBufSize = 1;                      /* Read buffer size                     */
    for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis              */
        sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
        if(sRet != 0){                  /*  In case of error                */
            printf("CCPU_FromBuf Error Md.23 [ret = %hd]\n", sRet);
            return sRet;
        }
        pusAxisErrNo[sLoop] = pusDataBuf[0];/* copy Axis error no. to variable     */
        ulOffset += AXIS_OFSET_SIZE;        /* Add the offset size per axis        */
    }

    /****************************************************************************/
    /* Md.24 Read the axis warning No.                                          */
    /****************************************************************************/
    ulOffset = AXSIS_WARNING_NO;        /* Offset address of the axis warning No.*/
    ulSize = 1;                         /* Read word size                       */
    ulBufSize = 1;                      /* Read buffer size                     */
    for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis              */
        sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
        if(sRet != 0){                  /*  In case of error                    */
            printf("CCPU_FromBuf Error Md.24 [ret = %hd]\n", sRet);
            return sRet;
        }
        pusAxisWarningNo[sLoop] = pusDataBuf[0];/* copy Axis warning no. to variable   */
        ulOffset += AXIS_OFSET_SIZE;            /* Add the offset size per axis        */
    }

    /****************************************************************************/
    /* Md.31 Read the status                                                    */
    /****************************************************************************/
    ulOffset = STATUS;                  /* Offset address of the status         */
    ulSize = 1;                         /* Read word size                       */
    ulBufSize = 1;                      /* Read buffer size                     */
    for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis              */
        sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
        if(sRet != 0){                  /*  In case of error                    */
            printf("CCPU_FromBuf Error Md.31 [ret = %hd]\n", sRet);
            return sRet;
        }
        pusStatus[sLoop] = pusDataBuf[0];   /* copy status to variable             */
        ulOffset += AXIS_OFSET_SIZE;        /* Add the offset size per axis        */
    }

    /****************************************************************************/
    /* Read the actual current value and servo status,                          */
    /* when RD75M or RD75MH is specified                                        */
    /****************************************************************************/
    if( (sType == UNITTYPE_RD75M) || (sType == UNITTYPE_RD75MH) ){  /* RD75M or RD75MH  */
        /************************************************************************/
        /* Md.101 Read the actual current value                                 */
        /************************************************************************/
        ulOffset = REAL_CURRENT_VAL;    /* Offset address of the actual current value */
        ulSize = 2;                     /* Read word size                       */
        ulBufSize = 2;                  /* Read buffer size                     */
        for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis          */
            sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
            if(sRet != 0){              /*  In case of Error                    */
                printf("CCPU_FromBuf Error Md.101 [ret = %hd]\n", sRet);
                return sRet;
            }
            plRealCurrentVal[sLoop] = (long)(pusDataBuf[0] + (pusDataBuf[1] << 16));  /* copy actual current value to variable */
            ulOffset += AXIS_OFSET_SIZE;                                              /* Add the offset size per axis          */
        }

        /************************************************************************/
        /* Md.108 Read the servo status                                         */
        /************************************************************************/
        ulOffset = SERVO_STATUS;        /* Offset address of the servo status   */
        ulSize = 1;                     /* Read word size                       */
        ulBufSize = 1;                  /* Read buffer size                     */
        for (sLoop = 0 ; sLoop < 4 ; sLoop++ ){ /* Read data of 4 axis          */
            sRet = CCPU_FromBuf(usIoNo / 16, ulOffset, ulSize, pusDataBuf, ulBufSize);
            if(sRet != 0){              /*  In case of error                    */
                printf("CCPU_FromBuf Error Md.108 [ret = %hd]\n", sRet);
                return sRet;
            }
            pusServoStatus[sLoop] = pusDataBuf[0];  /* Copy Servo status to variable */
            ulOffset += AXIS_OFSET_SIZE;            /* Add the offset size per axis        */
        }
    } 

    /****************************************************************************/
    /* Display information                                                      */
    /****************************************************************************/
    printf("/****************************************************************************/\n");
    printf("/************      RD75[I/O No.%3hXh] Information                  **********/\n",usIoNo);
    printf("/****************************************************************************/\n");     

    /****************************************************************************/
    /* Display axis No.                                                         */
    /****************************************************************************/
    printf("%35s%14s%14s%14s\n", "Axis1", "Axis2", "Axis3", "Axis4");

    /****************************************************************************/
    /* Md.20 Current feed value                                                 */
    /****************************************************************************/
    printf("CurrentFeedVal[Md.20 ]");
    printf("%13ld%14ld%14ld%14ld\n", plCurrentFeedVal[0], plCurrentFeedVal[1], plCurrentFeedVal[2], plCurrentFeedVal[3]);

    /****************************************************************************/
    /*  Display the actual current value, when RD75M or RD75MH is specified.    */
    /****************************************************************************/
    if( (sType == UNITTYPE_RD75M) || (sType == UNITTYPE_RD75MH) ){  /* RD75M or RD75MH  */
        /************************************************************************/
        /* Md.101 Actual current value                                          */
        /************************************************************************/
        printf("RealCurrentVal[Md.101]");
        printf("%13ld%14ld%14ld%14ld\n", plRealCurrentVal[0], plRealCurrentVal[1], plRealCurrentVal[2], plRealCurrentVal[3]);
    }

    /****************************************************************************/
    /* Md.23 Axis error No.                                                     */
    /****************************************************************************/
    printf("AxisErrNo     [Md.23 ]");
    printf("%13u%14u%14u%14u\n", pusAxisErrNo[0], pusAxisErrNo[1], pusAxisErrNo[2], pusAxisErrNo[3]);

    /****************************************************************************/
    /* Md.24 Axis warning No.                                                   */
    /****************************************************************************/
    printf("AxisWarningNo [Md.24 ]");
    printf("%13u%14u%14u%14u\n", pusAxisWarningNo[0], pusAxisWarningNo[1], pusAxisWarningNo[2], pusAxisWarningNo[3]);

    /****************************************************************************/
    /* Md.31 Status informations                                                */
    /****************************************************************************/
    printf("Status        [Md.31 ]");
    printf("%7s0x%04hX%8s0x%04hX%8s0x%04hX%8s0x%04hX \n", "", pusStatus[0], "", pusStatus[1], "", pusStatus[2], "", pusStatus[3]);

    /****************************************************************************/
    /*  Display the servo status, when RD75M or RD75MH is specified.            */
    /****************************************************************************/
    if( (sType == UNITTYPE_RD75M) || (sType == UNITTYPE_RD75MH) ){  /* RD75M or RD75MH  */
        /************************************************************************/
        /* Md.108 Servo status informations                                     */
        /************************************************************************/
        printf("ServoStatus   [Md.108]");
        printf("%7s0x%04hX%8s0x%04hX%8s0x%04hX%8s0x%04hX\n\n", "", pusServoStatus[0], "", pusServoStatus[1], "", pusServoStatus[2], "", pusServoStatus[3]);
    }

    return EXIT_SUCCESS;
}
