/****************************************************************************/
/* FILE NAME       MotionLink_R12.c                                         */
/* VERSION         1.00                                                     */
/* FUNCTION        Operation of Motion CPU using CCPU Function              */
/*                 (For R12CCPU-V)                                          */
/*                                                                          */
/*  Main function of sample program                                         */
/*  - MotionLinkMain                                                        */
/*    - The first argument is the case number of sample program to be       */
/*      executed                                                            */
/*    - The second argument specifies the parameters to be passed to the    */
/*      selected case number of sample program                              */
/*                                                                          */
/*  Configuration                                                           */
/*      This sample program requires following configuration                */
/*      ___________________                                                 */
/*      |  R  | R |   |   |                                                 */
/*      |  1  | 1 |   |   |                                                 */
/*      |  2  | 6 |   |   |   Unit1 iQ-R Platform compatible C Controller   */
/*      |  C  | M |   |   |   Unit2 iQ-R Platform compatible Motion CPU     */
/*      |  C  | T |   |   |                                                 */
/*      |  P  | C |   |   |                                                 */
/*      |  U  | P |   |   |                                                 */
/*      |     | U |   |   |                                                 */
/*      |     |   |   |   |                                                 */
/*      |_____|___|___|___|__                                               */
/*                                                                          */
/*  1. Motion Device Access(Read and Write)                                 */
/*     (MotionDeviceAccess())                                               */
/*     Repeat following steps 100 times every 100ms                         */
/*      - Write 1Word to the Motion CPU device D8000(Data Register)         */
/*      - Read 1Word from the Motion CPU device D8000                       */
/*      - Compare read data and write data                                  */
/*      - In case of data mismatch lit the USER LED                         */
/*                                                                          */
/*  2. Start SFC Motion Program                                             */
/*     (MotionSFCStarting())                                                */
/*      - Issue start request of the Motion CPU SFC program                 */
/*                                                                          */
/*  3. Execute Servo Program                                                */
/*    (MotionSVSTExec())                                                    */
/*     - Start request of the specified servo program with respect to the   */
/*       Axis no.1 of the Motion CPU                                        */
/*                                                                          */
/*  4. Modify present value                                                 */
/*    (MotionCHGAExec())                                                    */
/*    - Modify present value of Motion CPU Axis no.1                        */
/*                                                                          */
/*  5. Modify value of speed                                                */
/*    (MotionCHGVExec())                                                    */
/*    - Modify value of speed of Motion CPU Axis no.1                       */
/*                                                                          */
/*  6. Modify value of Torgue limit                                         */
/*    (MotionCHGTExec())                                                    */
/*    - Modify value of Torgue limit of Motion CPU Axis no.1                */
/*                                                                          */
/*  - Timeout value for this sample program is set to default(0:10sec)      */
/*                                                                          */
/* When you want to build this sample,                                      */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder        */
/* in C Controller module to C:\WindRiver\INCLUDE.                          */
/*                                                                          */
/*  Sample Program, user at own risk                                        */
/*                                                                          */
/* CREATED         :2014/10/22   MITSUBISHI ELECTRIC CORPORATION            */
/*                                                                          */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED   */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <usrLib.h>
#include    <taskLib.h>
#include    "CCPUFunc.h"

#define MOTIONCPU    0x3E1      /* Motion CPU:Unit 2 */

/*  Motion CPU Device Access (Read and Write)                          */
void MotionDeviceAccess();
/*  Start Motion CPU SFC Program                                       */
void MotionSFCStarting(int ProgramNo);
/*  Execute Servo Program                                              */
void MotionSVSTExec(int ProgramNo);
/*  Modify present value                                               */
void MotionCHGAExec(int iData);
/*  Modify value of Speed                                              */
void MotionCHGVExec(int iData);
/*  Modify value of Torgue limit                                       */
void MotionCHGTExec(int iData);

/****************************************************************************/
/* MODULE   : MotionLinkMain                                                */
/* ABSTRACT : Main function of Motion CPU operation using CCPU Function     */
/* FUNCTION : Start the task corresponding to the Sample Program            */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionLinkMain(
    int iNSampleNo,  /* Sample program no. to be executed                   */
    int iParam)      /* Parameter                                           */
{
    /************************************************************************/
    /* Create Task                                                          */
    /************************************************************************/
    switch(iNSampleNo){           /* Sample program case no. to be executed */
    case 1:                       /*     Sample program Case no.1           */
        /********************************************************************/
        /* Create Task 1(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionDeviceAccess,0,0,0,0,0,0,0,0,0);
        break;
    case 2:                             /* Sample program Case no.2          */
        /********************************************************************/
        /* Create Task 2(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionSFCStarting,iParam,0,0,0,0,0,0,0,0);
        break;
    case 3:                             /* Sample program Case no.3          */
        /********************************************************************/
        /* Create Task 3(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionSVSTExec,iParam,0,0,0,0,0,0,0,0);
        break;
    case 4:                             /* Sample program Case no.4          */
        /********************************************************************/
        /* Create Task 4(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionCHGAExec,iParam,0,0,0,0,0,0,0,0);
        break;
    case 5:                             /* Sample program Case no.5          */
        /********************************************************************/
        /* Create Task 5(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionCHGVExec,iParam,0,0,0,0,0,0,0,0);
        break;
    case 6:                             /* Sample program Case no.6          */
        /********************************************************************/
        /* Create Task 6(Use sp or taskSpawn)                      */
        /********************************************************************/
        sp((FUNCPTR)MotionCHGTExec,iParam,0,0,0,0,0,0,0,0);
        break;
    default:                             /* default case                     */
        /********************************************************************/
        /* Task not created                                        */
        /********************************************************************/
        break;
    }
}

/****************************************************************************/
/* MODULE   : MotionDeviceAccess                                            */
/* ABSTRACT : Read and Write access of Motion CPU device memory             */
/* FUNCTION : Read and Write Motion CPU(Unit 2)device D8000(Data Register)  */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionDeviceAccess()
{
    int             i;
    short           sRet;
    short           sControlData[2] = {0,0};
    short           sEndStatus[2] = {0,0};
    unsigned long   ulSize;
    unsigned long   ulBufSize;
    short  sWriteData;
    short  sReadData;

    /************************************************************************/
    /* Wait for 3sec                                                        */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Repeat Read and Write device memory access of Motion CPU             */
    /************************************************************************/
    printf("\nExecute Access to the Deveice of MotionCPU");
    printf("\n");
    ulSize      = 1;
    ulBufSize   = 1;
    sControlData[1]    = 1;
    for(i=0;i<100;i++){                                 /* Repeat 100 times */
        /********************************************************************/
        /* Write 1Word of Motion CPU (Unit 2) device D8000 (Data Register)  */
        /********************************************************************/
        sWriteData = i + 1;
        sRet = CCPU_DedicatedDInst("DDWR", MOTIONCPU, &sControlData[0], 2, &sWriteData, 1, (short*)"D8000", 5, &sEndStatus[0], 2, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
        if((sRet != 0)||(sControlData[0] != 0)){
            printf("\nCCPU_DedicatedDInst:DDWR Error!!\tErrorCode:%d[%04hxH]\tControlData:%d[%04hxH]",sRet,sRet,sControlData[0],sControlData[0]);
            break;
        }

        /********************************************************************/
        /* Wait 100msec                                                     */
        /********************************************************************/
        taskDelay(60 / 10);

        /********************************************************************/
        /* Read 1Word of Motion CPU (Unit 2) device D8000 (Data Register)   */
        /********************************************************************/
        sRet = CCPU_DedicatedDInst("DDRD", MOTIONCPU, &sControlData[0], 2, (short*)"D8000", 5, &sReadData, 1, &sEndStatus[0], 2, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
        if((sRet != 0)||(sControlData[0] != 0)){
            printf("\nCCPU_DedicatedDInst:DDRD Error!!\tErrorCode:%d[%04hxH]\tControlData:%d[%04hxH]",sRet,sRet,sControlData[0],sControlData[0]);
            break;
        }

        /********************************************************************/
        /*Compare Read and Write Data                                      */
        /********************************************************************/
        if(sWriteData != sReadData){  /* Read and Write data does not match */
            printf("\nData Compare Error!!\tCount %d",i+1);
            /****************************************************************/
            /* Lit USER LED                                   */
            /****************************************************************/
            CCPU_SetLEDStatus(0,1);
            break;
        }
        printf(".");
    }
    printf("\n---------------- End");
    printf("\n");
}

/****************************************************************************/
/* MODULE   : MotionSFCStarting                                             */
/* ABSTRACT : Starts Motion CPU SFC Program                                 */
/* FUNCTION : Execute Motion CPU(Unit2) SFC Program(Specify No.)            */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionSFCStarting(
    int ProgramNo)  /* Start Motion SFC Program No.(0-255)           */
{
    short   sRet;
    short   sEndBit[2];
    short   sEndStatus;

    /************************************************************************/
    /* Wait 3sec                                                            */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Start Motion CPU(Unit2) SFC Program (Specify No.)                    */
    /************************************************************************/
    sRet = CCPU_DedicatedDInst("SFCS", MOTIONCPU, (short*)&ProgramNo, 1, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
        printf("\nCCPU_DedicatedDInst:SFCS Success!!");
    }else{
        printf("\nCCPU_DedicatedDInst:SFCS Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
    }
    printf("\n\tProgramNo:%hd", (short)ProgramNo);
    printf("\n");

}

/****************************************************************************/
/* MODULE   : MotionSVSTExec                                                */
/* ABSTRACT : Starts Servo Program                                          */
/* FUNCTION : Start Motion CPU(Unit2) Axis no.1 Servo program               */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionSVSTExec(
    int ProgramNo)  /* Start Servo program No.(0-4095)                      */
{
    short   sRet;
    short   sEndBit[2];
    short   sEndStatus;

    /************************************************************************/
    /* Wait 3sec                                                            */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Start the Servo program with respect to the Axis no.1 of Motion CPU(Unit2) */
    /************************************************************************/
    sRet = CCPU_DedicatedDInst("SVST", MOTIONCPU, (short*)"J1", 2, (short*)&ProgramNo, 1, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
        printf("\nCCPU_DedicatedDInst:SVST Success!!");
    }else{
        printf("\nCCPU_DedicatedDInst:SVST Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
    }
    printf("\n\tProgramNo:%hd", (short)ProgramNo);
    printf("\n");

}

/****************************************************************************/
/* MODULE   : MotionCHGAExec                                                */
/* ABSTRACT : Modify present value                                          */
/* FUNCTION : Modify present value of Motion CPU (Unit2) Axis No.1          */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionCHGAExec(
    int iData)      /* Modify Present value(-2147483648 to 2147483647)      */
{
    short   sRet;
    short   sEndBit[2];
    short   sEndStatus;

    /************************************************************************/
    /* Wait 3sec                                                          */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Modify present value of Motion CPU(Unit2) Axis no.1                  */
    /************************************************************************/
    sRet = CCPU_DedicatedDInst("CHGA", MOTIONCPU, (short*)"J1", 2, (short*)&iData, 2, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
        printf("\nCCPU_DedicatedDInst:CHGA Success!!");
    }else{
        printf("\nCCPU_DedicatedDInst:CHGA Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
    }
    printf("\n\tlData:%d", iData);
    printf("\n");
}

/****************************************************************************/
/* MODULE   : MotionCHGVExec                                                */
/* ABSTRACT : Modify value of Speed                                         */
/* FUNCTION : Modify value of Speed of Motion CPU (Unit2) Axis no.1         */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionCHGVExec(
    int iData)      /* Modify value of speed(-2147483647 to 2147483647)     */
{
    short   sRet;
    short   sEndBit[2];
    short   sEndStatus;

    /************************************************************************/
    /* Wait 3sec                                                            */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Modify value of Speed of Motion CPU (Unit2) Axis no.1                */
    /************************************************************************/
    sRet = CCPU_DedicatedDInst("CHGV", MOTIONCPU, (short*)"J1", 2, (short*)&iData, 2, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
        printf("\nCCPU_DedicatedDInst:CHGV Success!!");
    }else{
        printf("\nCCPU_DedicatedDInst:CHGV Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
    }
    printf("\n\tlData:%d", iData);
    printf("\n");
}

/****************************************************************************/
/* MODULE   : MotionCHGTExec                                                */
/* ABSTRACT : Modify Torgue limit value                                     */
/* FUNCTION : Modify value of Torgue limit of Motion CPU (Unit2) Axis no.1  */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionCHGTExec(
    int iData)      /* Modify Torgue limit value(1-10000)                   */
{
    short   sRet;
    long    lData2;
    short   sEndBit[2];
    short   sEndStatus;

    /************************************************************************/
    /* Wait 3sec                                                            */
    /************************************************************************/
    taskDelay(60 * 3);

    /************************************************************************/
    /* Modify value of Torgue limit of Motion CPU (Unit2) Axis no.1         */
    /************************************************************************/
    lData2 = -1;
    sRet = CCPU_DedicatedDInst("CHGT", MOTIONCPU, (short*)"J1", 2, (short*)&iData, 2, (short*)&lData2, 2, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
        printf("\nCCPU_DedicatedDInst:CHGT Success!!");
    }else{
        printf("\nCCPU_DedicatedDInst:CHGT Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
    }
    printf("\n\tlData:%d", iData);
    printf("\n");
}
