/****************************************************************************/
/* FILE NAME       :MotionSyncInt_R12.c                                      */
/* VERSION         :1.00                                                     */
/* FUNCTION        :Handles the interrupt linked to the multi-CPU            */
/*                  synchronous interrupt with the Motion CPU                */
/*                  (For R12CCPU-V)                                          */
/*                                                                           */
/*  [Configuration]                                                          */
/*      This sample program requires the following configuration             */
/*      _______________                                                      */
/*      |  R  | R |   |                                                      */
/*      |  1  | 1 |   |                                                      */
/*      |  2  | 6 |   |                                                      */
/*      |  C  | M |   |   Unit1   :iQ-R Platform compatible C controller     */
/*      |  C  | T |   |   Unit2   :iQ-R Platform compatible Motion CPU       */
/*      |  P  | C |   |                                                      */
/*      |  U  | P |   |                                                      */
/*      |     | U |   |                                                      */
/*      |     |   |   |                                                      */
/*      |_____|___|___|__                                                    */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/*  In this sample program, read 32 points(4word) data from CPU buffer       */
/*  memory of Unit2 (U3E1G2048-U3E1G2079).                                   */
/*  When the interrupt is received the 32points read data is copied to       */
/*  its own unit of multi-CPU high speed communication area                  */
/*  (U3E0G2048-U3E0G2079)                                                    */
/*  In addition, a semaphore is also used to execute the work which cannot   */
/*  be executed in the ISR . This work will be executed in the task that has */
/*  received interrupt.                                                      */
/*                                                                           */
/*  When you check the operation in this sample program, make sure to        */
/*  operates a program for the return of the CPU access at the Unit2 CPU     */
/*  (U3E0G2048-U3E0G2079 -> U3E1G2048-U3E1G2079)                             */
/*                                                                           */
/* When you want to build this sample,                                       */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder         */
/* in C Controller module to C:\WindRiver\INCLUDE.                           */
/*                                                                           */
/*  Sample Program, user at own risk                                         */
/*                                                                           */
/* CREATED         :2014/10/22   MITSUBISHI ELECTRIC CORPORATION             */
/*                                                                           */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED    */
/*****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <semLib.h>
#include    <taskLib.h>
#include    "CCPUFunc.h"

#define MULTICPUSYNCINT 45      /* Multi-CPU Synchronous interrupt number:45 */
#define CCPU         0x3E0      /* C Language Controller:Unit 1 */
#define MOTIONCPU    0x3E1      /* Motion CPU:Unit 2 */

/* Semaphore for multi-synchronous interrupt delay processing            */
static SEM_ID semMotionSyncInt = NULL;

/* Multi-synchronous Interrupt handler                                   */
void MotionSyncIntISR(void);
/* Multi-synchronous Interrupt task                                      */
void MotionSyncIntTask(void);


/****************************************************************************/
/* MODULE   : MotionSyncIntMain                                             */
/* ABSTRACT : Multi-synchronous interrupt processing (sample program main)  */
/* FUNCTION : Start multi-synchronous interrupt processing task             */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionSyncIntMain(void)
{
    short           sRet;

    /************************************************************************/
    /* Register multi-synchronous interrupt handler                         */
    /************************************************************************/
    sRet = CCPU_EntryInt( MULTICPUSYNCINT, (CCPU_FUNCPTR)MotionSyncIntISR);
    /*  Close the bus                           */
    if(sRet != 0){
        printf("CCPU_EntryInt Error!!\tErrorCode:%d[%04hxH]\n",sRet,sRet);
        return;
    }

    /************************************************************************/
    /* Create semaphore for multi-synchronous interrupt delay processing    */
    /************************************************************************/
    semMotionSyncInt = semBCreate(SEM_Q_FIFO, SEM_EMPTY);
    if(semMotionSyncInt == NULL){
        return;
    }

    /************************************************************************/
    /* Start the task for multi-synchronous interrupt delay processing      */
    /************************************************************************/
    taskSpawn("tMSyncInt",100,0,0x1000,
                (FUNCPTR)MotionSyncIntTask,0,0,0,0,0,0,0,0,0,0);

    return;
}

/****************************************************************************/
/* MODULE   : MotionSyncIntISR                                              */
/* ABSTRACT : Multi-synchronous interrupt Handler                           */
/* FUNCTION : Execution of multi-synchronous interrupt                      */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionSyncIntISR(void)
{
    short           sRet;
    unsigned short  data[32];
    int             i;

    /************************************************************************/
    /* Read data from multi-CPU high speed communication area of unit2 CPU  */
    /************************************************************************/
    sRet = CCPU_FromBuf_ISR(MOTIONCPU,2048,32,data);
    if(sRet != 0){
        CCPU_DisableInt_ISR(MULTICPUSYNCINT);
        return;
    }

    /************************************************************************/
    /* User defined operation on the read data (below is dummy operation)   */
    /************************************************************************/
    for(i=0;i<32;i++)
        data[i]++;

    /************************************************************************/
    /* Write data to multi-CPU high speed communication area of self unit CPU */
    /************************************************************************/
    sRet = CCPU_ToBuf_ISR(CCPU,2048,32,data);
    if(sRet != 0){
        CCPU_DisableInt_ISR(MULTICPUSYNCINT);
        return;
    }

    /************************************************************************/
    /* Release semaphore for multi-synchronous interrupt delay processing   */
    /* so as to perform the work that cannot be executed in the ISR         */
    /************************************************************************/
    if((data[0] & 0x00FF) == 0){
        semGive(semMotionSyncInt);
    }

    return;
}

/****************************************************************************/
/* MODULE   : MotionSyncIntTask                                             */
/* ABSTRACT : Multi-synchronous interrupt Task                              */
/* FUNCTION : Execution of work that cannot be executed in the              */
/*            multi-synchronous interrupt handler                           */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void MotionSyncIntTask(void)
{
    short           sRet;
    STATUS          stat;
    long            velocity[4] = {0,50000,10000,25000};
    int             i;
    short           sEndBit[2];
    short           sEndStatus;

    /************************************************************************/
    /* Enable the registered multi-synchronous interrupt                    */
    /************************************************************************/
    sRet = CCPU_EnableInt(MULTICPUSYNCINT);
    if(sRet != 0){
        printf("CCPU_EnableInt Error!!\tErrorCode:%d[%04hxH]\n",sRet,sRet);
        return;
    }

    for(i=0;i<10000;i++){
        /********************************************************************/
        /* Waiting for the semaphore to be released from multi-synchronous  */
        /* interrupt handler                                                */
        /********************************************************************/
        stat = semTake(semMotionSyncInt,WAIT_FOREVER);
        if(stat == OK){
            /****************************************************************/
            /*  Work which cannot be executed in ISR, is executed here      */
            /****************************************************************/
            sRet = CCPU_DedicatedDInst("CHGV", MOTIONCPU, (short*)"J1", 2, (short*)&velocity[i%4], 2, &sEndBit[0], 2, &sEndStatus, 1, NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
            if((sRet == 0)&&(sEndBit[0] == 1)&&(sEndBit[1] == 0)){
                printf("CCPU_DedicatedDInst:CHGV Executed(%d[%04hxH])\t:velocity = %ld\n",sRet,sRet,velocity[i%4]);
            }
            else{
                printf("CCPU_DedicatedDInst:CHGV Error!!\tErrorCode:%d[%04hxH]\tEndStatus:%d[%04hxH]",sRet,sRet,sEndStatus,sEndStatus);
            }
        }
    }

    /************************************************************************/
    /* Remove/Un-register the multi-synchronous interrupt handler           */
    /************************************************************************/
    sRet = CCPU_EntryInt(MULTICPUSYNCINT,(CCPU_FUNCPTR)NULL);

    return;
}
