/****************************************************************************/
/* FILE NAME       :SDChange_R12.c                                         */
/* VERSION         :1.00                                                   */
/* FUNCTION        :When X0 is ON, Un-mount SD Card                        */
/*                  (For R12CCPU-V)                                        */
/*                                                                         */
/*  Sample Program to Un-mount the SD Card when device X0 is ON            */
/*                                                                         */
/*  Create and execute below task                                          */
/*  1. SDChangeMain()                                                      */
/*  -Open file, close, customized the access process as per                */
/*    the need or requirement                                              */
/*                                                                         */
/* When you want to build this sample,                                     */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder       */
/* in C Controller module to C:\WindRiver\INCLUDE.                         */
/*                                                                         */
/*  Sample Program, user at own risk                                       */
/*                                                                         */
/* CREATED         :2014/10/22   MITSUBISHI ELECTRIC CORPORATION           */
/*                                                                         */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED  */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <usrLib.h>
#include    <taskLib.h>
#include    "CCPUFunc.h"

#define BIT_ON          1   /* BIT ON value                               */
#define BIT_OFF         0   /* BIT OFF value                              */
#define CCPU_SDRIVE_SD  1   /* SD Card                                    */

FILE*   gpFpSD;             /* File Pointer                               */

/* SD Card Change processing                                              */
void    SDChange();

/****************************************************************************/
/* MODULE   : SDChangeMain                                                  */
/* ABSTRACT : Create SD Card exchange task                                  */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void SDChangeMain()
{
    /************************************************************************/
    /* Create Task(Use sp or taskSpawn)                                     */
    /************************************************************************/
    sp((FUNCPTR)SDChange, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : SDChange                                                      */
/* ABSTRACT : SD Card exchange processing                                   */
/* FUNCTION : When the state change is detected, Un-mount the SD Card       */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void SDChange()
{
    short           sRet;
    long            plInfo[4];
    unsigned short  usData;
    unsigned long   ulBufSize;

    /************************************************************************/
    /* - File Open                                                          */
    /*  Read and Write files when the state of SD card is mounted           */
    /*  Open file on the SD card in binary mode to read and write           */
    /************************************************************************/
    /*                        User Processing                               */
    /*  gpFpSD = fopen("/2/sample_SD.dat", "rb+");                          */

    /************************************************************************/
    /*  Read X0 state every second, Un-mount SD card if X0 is turned ON     */
    /************************************************************************/
    while(1){   /* Infinite Loop                                            */
        /********************************************************************/
        /* Read the value of X0                                             */
        /********************************************************************/
        sRet = CCPU_X_In_BitEx(0, 0, &usData);
        if(sRet != 0){
            printf("\nX0 Read Error!! sRet = %hd ( %hxH )\n", sRet, sRet);
            return;
        }

        /********************************************************************/
        /*  When X0 is ON, stop the file access such as fread/fwrite        */
        /*  Un-mount SD Card                                                */
        /********************************************************************/
        if(usData == BIT_ON){                   /*   X0 is ON              */
            /****************************************************************/
            /* - File Close                                                 */
            /*  Stop the file access of SD Card                             */
            /****************************************************************/
            /*                        User Processing                       */
            /* fclose(gpFpSD);                                              */

            /****************************************************************/
            /*  Un-mount the SD Card                                        */
            /****************************************************************/
            sRet = CCPU_UnmountMemoryCard(CCPU_SDRIVE_SD);
            if(sRet != 0){
                printf("\nCCPU_UnmountMemoryCard Error!! sRet = %hd ( %hxH )\n", sRet, sRet);
                return;
            }
            break;
        }else{                              /*    X0 is OFF                 */
        /********************************************************************/
        /* - File Access Processing                                         */
        /*  When X0 is OFF, access files on SD card using fread/fwrite      */
        /*                                                                  */
        /********************************************************************/
        /*                        User Processing                           */
        }

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    printf("SD Card Installed (Un-mounted) state\n");

    /************************************************************************/
    /*  After confirmation that SD Card state is Un-mount                   */
    /*  Turn OFF X0                                                         */
    /************************************************************************/
    /*                        User Processing                               */

#if 0   /*[Start] Mount SD Card without detaching the SD card               */
    /************************************************************************/
    /*  The following procedure is for re-mounting the SD card without      */
    /*  detaching the SD card. If SD Card is detached and attached again,   */
    /*  the following processing is unnecessary because mounting of SD Card */
    /*  will be done automatically.                                         */
    /************************************************************************/
    /*  Execute the mounting of SD Card                                     */
    /************************************************************************/
    sRet = CCPU_MountMemoryCard(CCPU_SDRIVE_SD);
    if(sRet != 0){
        printf("\nCCPU_MountMemoryCard Error sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }
#endif  /*[End] Mount SD Card without detaching the SD card                 */

    /************************************************************************/
    /*  Read the status information of the C Controller module every 1sec   */
    /*  If the SD card status is mounted, file on the SD card can be        */
    /*  accessed                                                            */
    /************************************************************************/
    while(1){  /* Infinite Loop                                             */
        /********************************************************************/
        /*  Read the state information of the C controller module           */
        /********************************************************************/
        ulBufSize = sizeof(plInfo) / 4;
        sRet = CCPU_GetCpuStatus(plInfo, ulBufSize);
        if(sRet != 0){
            printf("\nCCPU_GetCpuStatus Error!! sRet = %hd ( %hxH )\n", sRet, sRet);
            return;
        }

        /********************************************************************/
        /*  Check if the SD Card is Mounted                                 */
        /********************************************************************/
        if((plInfo[1] & 0x0018) == 0){ /* SD Card Mounted State?          */
            printf("SD Card mounted(mount)state\n");
            /****************************************************************/
            /*  Open file on the SD Card                         */
            /*  File access enable                               */
            /****************************************************************/
            /*                   User Processing                            */
            break;
        }

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    return; 
}
