/****************************************************************************/
/* FILE NAME       DevAccessChanRBF_R12.c                                   */
/* VERSION         1.00                                                     */
/* FUNCTION        MELSEC Data Link Function(MD Function)                   */
/*                 (Sample Program for R12CCPU-V)                           */
/*                                                                          */
/*  Device Access via CPU Bus                                               */
/*                                                                          */
/*  Contents                                                                */
/*  - CPU_BusAccessRtnMain                                                  */
/*    R series bus interface(channel No.12) Multi-CPU configuration is used,*/
/*    when device M0 is ON then read device D0 of CPU No.1 and write to     */
/*    device D1.                                                            */
/*    Then read M0 to M15, D0, D1 and CPU shared memory(SPB1)               */
/*                                                                          */
/*      ____________                                                        */
/*      |R | C  |  |                                                        */
/*      |C | C  |  |                                                        */
/*      |P | P  |  |                                                        */
/*      |U | U  |  |                                                        */
/*      |  |    |  |                                                        */
/*      |  |    |  |                                                        */
/*      |__|____|__|__                                                      */
/*        |_|                                                               */
/*          Channel 12                                                      */
/*                                                                          */
/* When you want to build this sample,                                      */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder        */
/* in C Controller module to C:\WindRiver\INCLUDE.                          */
/*                                                                          */
/*  Sample Program, user at own risk                                        */
/*                                                                          */
/* CREATED         : 2014/10/22   MITSUBISHI ELECTRIC CORPORATION           */
/*                                                                          */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED   */
/****************************************************************************/

#include    <vxWorks.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <usrLib.h>
#include    <taskLib.h>
#include    "MDFunc.h"

#define OWN_STATION     0xFF    /* Station No.(Local Staion)                */

/* R series bus interface processing                                        */
void CPU_BusAccessRtn();

/****************************************************************************/
/* MODULE   : CPU_BusAccessRtnMain                                          */
/* ABSTRACT : R series bus interface task                                   */
/* FUNCTION : Start the task                                                */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CPU_BusAccessRtnMain()
{
    /************************************************************************/
    /* Create Task(Use sp or taskSpawn)                                     */
    /************************************************************************/
    sp((FUNCPTR)CPU_BusAccessRtn, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    return;
}

/****************************************************************************/
/* MODULE   : CPU_BusAccessRtn                                              */
/* ABSTRACT : R series bus interface                                        */
/* FUNCTION : R series bus interface processing                             */
/* NOTE     : NONE                                                          */
/* RETURN   : NONE                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void CPU_BusAccessRtn()
{
    short   sRet;
    long    lPath;
    short   sData;
    long    lSize;
    long    plDev[7];
    short   psBuf[3];

    /************************************************************************/
    /* Open R series bus interface                                          */
    /************************************************************************/
    sRet = mdOpen(12, -1, &lPath); 
    if(sRet != 0){
        printf("\nmdOpen Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        return;
    }

    /************************************************************************/
    /* Read device D0 of CPU1                                               */
    /************************************************************************/
    lSize = 2;
    sRet = mdReceiveEx(lPath, 0, 1, DevD, 0, &lSize, &sData);
    if(sRet != 0){
        printf("\nmdReceiveEx(D0)Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Write the read data of device D0 to device D1                        */
    /************************************************************************/
    sRet = mdSendEx(lPath, 0, 1, DevD, 1, &lSize, &sData);
    if(sRet != 0){
        printf("\nmdSendEx Error sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* M0 ON                                                           */
    /************************************************************************/
    sRet = mdDevSetEx(lPath, 0, 1, DevM, 0);
    if(sRet != 0){
        printf("\nmdDevSetEx Error sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Read the value of device M0-M15, D0, D1                              */
    /************************************************************************/
    plDev[0] = 2;     /*Number of Blocks                                    */
    plDev[1] = DevM;  /*Block No.1 Device Type                              */
    plDev[2] = 0;     /*Block No.1 Head Device No.                          */
    plDev[3] = 16;    /*Block No.1 Read bits                                */
    plDev[4] = DevD;  /*Block No.2 Device Type                              */
    plDev[5] = 0;     /*Block No.2 Head Device No.                          */
    plDev[6] = 2;     /*Block No.2 Read bits                                */
    sRet = mdRandREx(lPath, 0, 1, plDev, psBuf, 6);
    if(sRet != 0){
        printf("\nmdRandREx Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Read CPU shared memory (SPB1) of CPU1                                */
    /************************************************************************/
    sRet = mdReceiveEx(lPath, 0, OWN_STATION, DevSPB1, 0, &lSize, &sData);
    if(sRet != 0){
        printf("\nmdReceiveEx Error  sRet = %hd ( %hxH )\n", sRet, sRet);
        mdClose(lPath);
        return;
    }

    /************************************************************************/
    /* Print the read data of device M0-M15, D0, D1                         */
    /************************************************************************/
    printf("M0-M15    = %hd ( %hxH )\n", psBuf[0], psBuf[0]);
    printf("D0        = %hd ( %hxH )\n", psBuf[1], psBuf[1]);
    printf("D1        = %hd ( %hxH )\n", psBuf[2], psBuf[2]);

    /************************************************************************/
    /* Print the read data of CPU shared memory                             */
    /************************************************************************/
    printf("SPB1 0000 = %hd ( %hxH )\n", sData, sData);

    /************************************************************************/
    /* Close R series bus interface                                         */
    /************************************************************************/
    mdClose(lPath);
    return;
}
