/****************************************************************************/
/* FILE NAME       FTPGet_R12.c                                            */
/* VERSION         1.00                                                    */
/* FUNCTION        : Access PC configured as server (FTP server) using FTP */
/*                   (For R12CCPU-V)                                       */
/*                                                                         */
/*  Outline -                                                              */
/*  1. This program downloads the program file from the PC server          */
/*     (FTP server), and, create and activate tasks                        */
/*  2. After above step, transmits the file to PC (FTP Server) from        */
/*     C Controller module                                                 */
/*                                                                         */
/*  Pre-requisites -                                                       */
/*  1. Two out files are created from this sample program                  */
/*      1) FTPGet.out (for the sample program execution)                   */
/*      2) Demo1.out  (for download, create this file by copying)          */
/*                                                                         */
/*  2. Locate the file for download in the FTP server                      */
/*  Note: Set zero in the empty slot in the I/O assignment setting of the  */
/*        setting/monitoring tool                                          */
/*                                                                         */
/*  Procedure -                                                            */
/*  Spawn the following functions as a task using STARTUP file             */
/*                                                                         */
/*  Example of STARTUP.CMD (Upload this file on SD)                        */
/*  1)When user use SD card                                                */
/*      ld(1,0,"/2/FTPGet.out")                                            */
/*      taskSpawn("initsk", 150, 0, 8192, func, 0)                         */
/*                                                                         */
/*  2)When user use RAM disk                                               */
/*      ld(1,0,"/2/FTPGet.out")                                            */
/*      taskSpawn("initsk_RAM", 150, 0, 8192, func_ram, 1024*2)            */
/*                                                                         */
/* When you want to build this sample,                                     */
/* Please copy CCPUFunc.h and MDFunc.h from SYSTEMROM INCLUDE folder       */
/* in C Controller module to C:\WindRiver\INCLUDE.                         */
/*                                                                         */
/* Sample Program, User at Own Risk                                        */
/*                                                                         */
/* CREATED         2014/10/22   MITSUBISHI ELECTRIC CORPORATION            */
/*                                                                         */
/* COPYRIGHT (C) 2014 MITSUBISHI ELECTRIC CORPORATION ALL RIGHTS RESERVED  */
/****************************************************************************/
#include    <vxworks.h>
#include    <stdio.h>
#include    <ftpLib.h>
#include    <stat.h>
#include    <usrLib.h>
#include    <loadLib.h>
#include    <ioLib.h>
#include    <symLib.h>
#include    <string.h>
#include    <selectLib.h>
#include    <dosFsLib.h>
#include    <xbdRamDisk.h>
#include    "CCPUFunc.h"

#define RAM_DRV_NAME    "/RAMDISK"  /* RAM Disk Name(Optional)          */

extern  SYMTAB_ID       sysSymTbl;  /* symbol(function name)                */

/* Create RAM disk                                                          */
int ramMake(int iBlockSize);        /* MakeRAMDisk_R12.c                    */
/* For Monitoring Task Operation                                            */
void vY_Output1(void);
/* Create RAM disk (spawn function as task)                                 */
void func_ram(int iBlockSize);
/* FTP access                                                               */
void func(int iFlg);
/* FTP file acquisition                                                     */
int tcpFileGet(int iFlg);
/* FTP file transmission                                                    */
void tcpFilePut(int iFlg);

/****************************************************************************/
/* MODULE   : ramMake                                                       */
/* ABSTRACT : Create RAM disk device                                        */
/* FUNCTION : Creates RAM disk of the specified number of blocks            */
/* NOTE     : None                                                          */
/* RETURN   :                                                               */
/*            0             Normal                                          */
/*            Otherwise     Error                                           */
/* CREATE   : 2014-10-14    MITSUBISHI ELECTRIC CORPORATION                 */
/****************************************************************************/
int ramMake(
    int iBlockSize)       /* Number of blocks of RAM disk                   */
{
    int             iRamDiskSize;
    device_t        stxbdId;

    /************************************************************************/
    /* Create RAM disk                                                      */
    /************************************************************************/
    iRamDiskSize = (512 * iBlockSize);  /* RAM disk size                    */

    /************************************************************************/
    /* Create RAM disk(device)                                              */
    /************************************************************************/
    stxbdId = (device_t)(BLK_DEV*)xbdRamDiskDevCreate(512, iRamDiskSize, 0, RAM_DRV_NAME);
    if(stxbdId == NULLDEV) {
        goto ERROR_EXIT;
    }

    /************************************************************************/
    /* Configuring file-system                                              */
    /************************************************************************/
    /* Format RAM disk                                                      */
    dosFsVolFormat(RAM_DRV_NAME, (DOS_OPT_BLANK | DOS_OPT_QUIET), NULL);

    return OK;

/****************************************************************************/
/* error function                                                           */
/****************************************************************************/
ERROR_EXIT:
    printf("\nramMake ERROR: iBlockSize=%d\n", iBlockSize);
    return ERROR;
}

/****************************************************************************/
/* MODULE   : vY_Output1                                                    */
/* ABSTRACT : For Monitor Task Operation                                    */
/* FUNCTION : Increment Bits Y0-YF every second                             */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void vY_Output1()
{
    short           sRet;
    unsigned short  pusData[1];

    /************************************************************************/
    /*  Increment Bits Y0-YF every one second                               */
    /************************************************************************/
    pusData[0] = 0;
    while(1){   /* Infinite Loop                                            */
        /********************************************************************/
        /* Write Y0 - YF                                                    */
        /********************************************************************/
        sRet = CCPU_Y_Out_WordEx(0, 0, 1, pusData, 0);
        if(sRet != 0){
            printf("CCPU_Y_Out_WordEx Error sRet = %hd ( %hxH )\n", sRet, sRet);
            break;
        }

        /********************************************************************/
        /* Increment output data                                            */
        /********************************************************************/
        pusData[0]++;

        /********************************************************************/
        /* Wait 1sec                                                        */
        /********************************************************************/
        taskDelay(60);
    }

    return;
}

/****************************************************************************/
/* MODULE   : func_ram                                                      */
/* ABSTRACT : Create RAM disk device                                        */
/* FUNCTION : Create RAM disk device                                        */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void func_ram(
    int iBlockSize)    /* Number of RAM disk blocks                         */
{
    int sRet;

    /************************************************************************/
    /* Create RAM disk                                                      */
    /************************************************************************/
    sRet = ramMake(iBlockSize);
    if(sRet != OK) {
        return;
    }

    /************************************************************************/
    /* Execute TEST (on RAM disk)                                           */
    /************************************************************************/
    func(1);
}

/****************************************************************************/
/* MODULE   : func                                                          */
/* ABSTRACT : FTP access process                                            */
/* FUNCTION : Download the file from PC FTP server, spawn the task          */
/*            to monitor FTP access process and transmit file from          */
/*            C Controller to PC (FTP Server)                               */
/* NOTE     : None                                                          */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void func(
    int iFlg)   /* Destination of the file to be saved (0: SD, others: RAM disk) */
{
    STATUS      stStatus= OK;
    int         iResult; 
    MODULE_ID   stModuleId;
    SYMBOL_DESC stSymbolDesc;

    /************************************************************************/
     /* Download the file from FTP server and save on C Controller module   */
    /************************************************************************/
    if(tcpFileGet(iFlg) != OK) {
        return;
    }

    /************************************************************************/
    /* Load the received program file into RAM workspace of C               */
    /* Controller module                                                    */
    /* Example: The file name on FTP server is Demo1.out                    */
    /************************************************************************/
    if(iFlg != 0) { /* When user use RAM disk                               */
        /* RAM disk name "/RAMDISK"                                         */
        stModuleId = (MODULE_ID)ld(1,0,(char*)"/RAMDISK/Demo1.out");
    } else {       /* When user use SD card                                 */
        stModuleId = (MODULE_ID)ld(1,0,(char*)"/2/Demo1.out");
    }

    /************************************************************************/
    /* If module loading failed stop further processing                     */
    /************************************************************************/
    if(stModuleId == NULL){
        return;
    }

    /************************************************************************/
    /* Retrieve the symbol by name of the task monitor function from the    */
    /* module (program file) loaded and spawn the monitor task              */ 
    /*                                                                      */
    /* Note that an underscore is prefixed in the retrieved symbol name.    */
    /************************************************************************/
    memset (&stSymbolDesc, 0, sizeof(SYMBOL_DESC));
    stSymbolDesc.mask = SYM_FIND_BY_NAME;
    stSymbolDesc.name = "vY_Output1";
    stStatus = symFind( sysSymTbl, &stSymbolDesc );
    if(stStatus == OK) {
      iResult = taskSpawn((char *)"TaskName1", (int)150, (int)0, (int)1024, (FUNCPTR)stSymbolDesc.value, 0,0,0,0,0,0,0,0,0,0);
    }

    /************************************************************************/
    /*  Transmit file from C Controller module to FTP server                */
    /************************************************************************/
    tcpFilePut(iFlg);
}

/****************************************************************************/
/* MODULE   : tcpFileGet                                                    */
/* ABSTRACT : FTP file Receive process                                      */
/* FUNCTION : Receive file from PC server via FTP                           */
/* NOTE     : If reception failed error LED is lit                          */
/* RETURN   :                                                               */
/*            0             Normal                                          */
/*            otherwise     Error                                           */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
int tcpFileGet(
    int iFlg)   /* Destination of the file to be saved (0: SD, others: RAM disk) */
{
    /************************************************************************/
    /* FTP server function is required to operate on the PC server side.    */
    /* The IP address, user-name, password, and account of FTP server       */
    /* should be changed properly.                                          */
    /* Free software (Tiny FTP Deamon/95)+XP Pro of FTP server is used for  */
    /* confirmation.                                                        */
    /************************************************************************/

    int     iCtrlSock;
    int     iDataSock;
    char    pcBuf [2048];
    int     iNByte = -1;
    STATUS  stStatus= OK;
    FILE*   pFd;
    short   sRet;
    int     i;
    int     iNRet;
    fd_set  st_fdset_r;
    struct timeval  stTimeout;

    /************************************************************************/
   /* Retry FTP transfer processing up to 3 times                           */
    /************************************************************************/
    for(i=0; i<3; i++) {                /* Retry                            */
        /********************************************************************/
        /* Start FTP transfer                                               */
        /* Match FTP command to specifications of FTP server used           */
        /* Argument "RETR" passed is a file acquisition command)            */
        /********************************************************************/
        if (ftpXfer ("192.168.3.3", "target", "password", "",
            "RETR %s", "", "Demo1.out",  /* Demo1.out is a file to be acquired   */
            &iCtrlSock, &iDataSock) == ERROR) {
            /* If the socket is not created or the connection is not established */
            /****************************************************************/
            /* If there is time-out error, TCP connection established with   */
            /* the FTP server remains intact                                 */
            /****************************************************************/

            /* retry                                                        */
            continue;   
        } else { /* Socket is created and connection is established successfully */
            /* Stop retry                                                   */
            break;
        }
    }

    /************************************************************************/
     /* If retry count is 3 then error is return                            */
    /************************************************************************/
    if(i >= 3) {    /* Is retry 3 times?                                    */
        /* Blink USER LED (RED)                                             */
        sRet = CCPU_SetLEDStatus(0, 1);
        /* Return error                                                     */
        return ERROR;
    }
    /* Get the content of the file from the acquired data socket            */

    /************************************************************************/
    /* Open the file to saved on C Controller module                        */
    /************************************************************************/
    if(iFlg != 0) {             /* When user use RAM disk                   */
        pFd = fopen("/RAMDISK/Demo1.out","wb");
    } else {                    /* When user use SD card                    */
        pFd = fopen("/2/Demo1.out","wb");
    }

    /************************************************************************/
    /* When file open is successful (File is received successfully via FTP) */
    /************************************************************************/
    if(pFd != 0) {
        FD_ZERO(&st_fdset_r);
        FD_SET(iDataSock,&st_fdset_r);
        stTimeout.tv_sec = 30;
        stTimeout.tv_usec = 0;
        do{
            iNRet = select(iDataSock+1,&st_fdset_r,NULL,NULL,&stTimeout);
            if((iNRet == 0) || (iNRet == ERROR)){
                stStatus = ERROR;
                break;
            }
            iNByte = read (iDataSock, pcBuf, sizeof (pcBuf));
            if(iNByte <= 0){
                break;
            }
            if(iNByte > fwrite(pcBuf, 1, iNByte, pFd)){
                stStatus = ERROR;
                break;
            }
        }while(1);
        fclose(pFd);
    } else {
        stStatus = ERROR;
    }
    close (iDataSock);

    /************************************************************************/
    /*  File Read Error                                                     */
    /************************************************************************/
    if (iNByte < 0) { 
        stStatus = ERROR;
    }

    /************************************************************************/
    /* Confirm FTP Communication completed                                  */
    /************************************************************************/
    if (ftpReplyGet (iCtrlSock, TRUE) != FTP_COMPLETE) {
        stStatus = ERROR;
    }

    /************************************************************************/
   /* Transmit FTP command  --Logout of FTP communication and disconnect--  */
    /************************************************************************/
    if (ftpCommand (iCtrlSock, "QUIT", 0, 0, 0, 0, 0, 0) != FTP_COMPLETE) {
        stStatus = ERROR;
    }
    close (iCtrlSock);

    /************************************************************************/
    /* Return OK                                                            */
    /************************************************************************/
    if(stStatus == OK) {
        return OK;
    }

    /************************************************************************/
    /* In case of error Blink USER LED                                      */
    /************************************************************************/
    sRet = CCPU_SetLEDStatus(0, 2);     /* Blink USER LED (RED)             */
    return ERROR;                       /* Return error                     */
}

/****************************************************************************/
/* MODULE   : tcpFilePut                                                    */
/* ABSTRACT : FTP file transmit process                                     */
/* FUNCTION : The file is transmit to PC server via FTP                     */
/* NOTE     : If transmission failed error LED is lit                       */
/* RETURN   : None                                                          */
/* CREATE   : 2014-10-22    MITSUBISHI ELECTRIC CORPORATION                 */
/* UPDATE   :                                                               */
/****************************************************************************/
void tcpFilePut(
    int iFlg)   /* Destination of the file to be saved (0: SD, others: RAM disk) */
{
    /************************************************************************/
    /* FTP server function is required to operate on the PC server side.    */
    /* The IP address, user-name, password, and account of FTP server       */
    /* should be changed properly.                                          */
    /* Free software (Tiny FTP Deamon/95)+XP Pro of FTP server is used for  */
    /* confirmation                                                         */
    /************************************************************************/

    int     iCtrlSock;
    int     iDataSock;
    char    pcBuf [2048];
    int     iNByte = 0;
    STATUS  stStatus= OK;
    FILE*   pFd;
    short   sRet;
    int     i;

    /************************************************************************/
    /* Retry FTP transfer processing up to 3 times                          */
    /************************************************************************/
    for(i=0; i<3; i++) {                            /* Retry 3 times        */
        /********************************************************************/
        /* Start FTP transfer.                                              */
        /* Match FTP command to specifications of FTP server used.          */
        /* Argument "STOR" is a file transmission command                   */
        /* Example:Transmit the file to FTP server as Demo1_copy.out.       */
        /********************************************************************/
        if (ftpXfer ("192.168.3.3", "target", "password", "",
            "STOR %s", "", "Demo1_copy.out", /* "Demo1_copy.out" specified on C Controller module is the name on FTP server  */
            &iCtrlSock, &iDataSock) == ERROR) {
            /* If the socket is not created or the connection is not established */
            /* Retry                                                             */
            continue;   
        } else { /* socket is created, and connection is established successfully */
            /* Stop Retry                                                         */
            break;
        }
    }

    /************************************************************************/
    /* If retry count is 3 then error is return                             */
    /************************************************************************/
    if(i >= 3) {
        sRet = CCPU_SetLEDStatus(0, 1);     /* Blink USER LED (RED)         */
        return;                             /* Return error                 */
    }
    /* Get the content of the file from the acquired data socket            */

    /************************************************************************/
    /* Open the file saved on C Controller module                           */
    /************************************************************************/
    if(iFlg != 0) {         /* When you use RAM disk                        */
        pFd = fopen("/RAMDISK/Demo1.out","rb");
    } else {                /* When you use SD card                         */
        pFd = fopen("/2/Demo1.out","rb");
    }

    /************************************************************************/
    /* File Transmission                                                    */
    /************************************************************************/
    if(pFd == NULL) {       /* File to transmit cannot open                 */
        stStatus = ERROR;
    }else{                  /* File to transmit open successfully           */
        /********************************************************************/
        /* Read the content of file and write the same on data socket       */
        /********************************************************************/
        while ((iNByte = fread( pcBuf, 1, sizeof (pcBuf), pFd)) > 0)
        {
            if(write(iDataSock, pcBuf, iNByte)== ERROR)
            {
                stStatus = ERROR;
                break;
            }
        }
        fclose(pFd);
    }
    close (iDataSock);

    /************************************************************************/
    /*  File Reading Failed                                                 */
    /************************************************************************/
    if (iNByte < 0) {
        stStatus = ERROR;
    }

    /************************************************************************/
    /* Confirm the completion of FTP communication                          */
    /************************************************************************/
    if (ftpReplyGet (iCtrlSock, TRUE) != FTP_COMPLETE) {
        stStatus = ERROR;
    }

    /************************************************************************/
    /* Transmit FTP command  --Logout of FTP communication and disconnect-- */
    /************************************************************************/
    if (ftpCommand (iCtrlSock, "QUIT", 0, 0, 0, 0, 0, 0) != FTP_COMPLETE) {
        stStatus = ERROR;
    }
    close (iCtrlSock);

    /************************************************************************/
    /* In case of error Blink USER LED                                      */
    /************************************************************************/
    if(stStatus == ERROR) {
        sRet = CCPU_SetLEDStatus(0, 2);    /* Blink USER LED (RED)          */
    }
}
